import { Component, OnInit } from '@angular/core';
import { ActivatedRoute, Router } from '@angular/router';
import { FormBuilder, FormGroup, Validators, FormControl } from '@angular/forms';
import { HttpClient } from '@angular/common/http';
import { DataService } from 'src/app/services/data.service';
import { ActionsService } from 'src/app/services/actions.service';
import { ModalController } from '@ionic/angular';

@Component({
  selector: 'app-vaccination',
  templateUrl: './vaccination.page.html',
  styleUrls: ['./vaccination.page.scss'],
})
export class VaccinationPage implements OnInit {
  public action: string = '';
  public actionID: number = 0;
  public actionType: string = '';
  public vaccinationForm: FormGroup;
  public isVaccineAvailable = false;
  public isVaccineOther = false;
  public filteredVaccines: any = [];
  public vaccineSelected: any;

  // Set variable related to if the Vaccination page is being served normally from the Actions page or from the Custom Form page as a modal
  public isModal = false;

  public routes = [
    { id: 1, name: 'Subcutaneously (SQ)' },
    { id: 2, name: 'Intramuscularly (IM)' },
    { id: 3, name: 'Intravenously (IV)' },
    { id: 4, name: 'As an epidural' },
    { id: 5, name: 'Fat pad behind ear' },
    { id: 6, name: 'Each Nostril' },
    { id: 7, name: 'Eye' },
    { id: 8, name: 'Left Eye' },
    { id: 9, name: 'Right Eye' },
    { id: 10, name: 'Intradermally' },
    { id: 11, name: 'Intramammary' },
    { id: 12, name: 'Intranasally' },
    { id: 13, name: 'Intrauterine' },
    { id: 14, name: 'Orally' },
    { id: 15, name: 'Paravertebrally' },
    { id: 16, name: 'Rectally' },
    { id: 17, name: 'Topically' }
  ];

  // 3 = Vaccination Event; Reference database `action_types` table
  private actionTypeID = 3;
  private itemType: string = '';
  private selected: any;
  private selectedAmount: string = '';
  private vaccineIDs: any = [];
  private vaccines: any = [];

  // Set variable related to if the Health page is being served normally from the Actions page or from the Custom Form page as a modal
  private vaccineOtherName: any;
  private selectedRoute: any;

  constructor(
    public formBuilder: FormBuilder,
    private route: ActivatedRoute,
    private router: Router,
    private httpClient: HttpClient,
    private dataService: DataService,
    private actionsService: ActionsService,
    private modalCtrl: ModalController
  ) {
    this.route.queryParams.subscribe(_ => {
      if (this.router.getCurrentNavigation()) {
        // `animals` or `groups`
        this.itemType = this.router.getCurrentNavigation()?.extras.state?.['itemType'];
        // e.g., `vaccination`
        this.actionType = this.router.getCurrentNavigation()?.extras.state?.['actionType'];
        // e.g., `vaccination`
        this.action = this.router.getCurrentNavigation()?.extras.state?.['action'];
        // animals or groups selected by the user
        this.selected = this.router.getCurrentNavigation()?.extras.state?.['selected'];
        // `single` or `multiple`
        this.selectedAmount = this.router.getCurrentNavigation()?.extras.state?.['selectedAmount'];
      }
    });

    this.vaccinationForm = formBuilder.group({
      date: new FormControl('', Validators.required),
      volume: new FormControl(''),
      units: new FormControl('', Validators.required),
      vaccine: new FormControl('', Validators.required),
      vaccineOther: new FormControl(''),
      route: new FormControl('', Validators.required),
      serialNumber: new FormControl(''),
      expirationDate: new FormControl('')
    });
  }

  // When the user changes the Route of Administration, set variable to keep track of selected route
  routeChange(event: any) {
    if (event) {
      const filteredRoute = this.routes.filter((r: any) => r.id === +event.detail.value);
      if (filteredRoute) {
        this.selectedRoute = filteredRoute[0]?.name;
      }
    }
  }

  // If the user taps "Close", the modal will be dismissed
  dismiss() {
    this.modalCtrl.dismiss();
  }

  /*
  * If the user taps "Done" (required data entered), continue with processing the form
  * Determine if the Vaccination page is being accessed via a modal from the Custom Form page or normally through Actions
  */
  done(action: string) {
    // Initialize empty dictionary and set form equal to current vaccination form data
    let formDictionary: any = {};
    let form: any = this.vaccinationForm;

    // Add the key/value pairings to the dictionary
    Object.keys(form.controls).forEach(key => {
      formDictionary[key] = form.get(key).value;
    });

    if (formDictionary) {
      // example: (3, 'vaccination', 'animals', [31], 'single', {formData})

      /*
      * If the Vaccination page was navigated to via modal from the Actions Custom Processing Form
      * Dismiss the modal and return the formDictionary
      */
      if (this.isModal) {
        if (this.isVaccineOther) {
          this.vaccineOtherName = this.vaccinationForm.controls['vaccineOther'].value.trim();
          this.modalCtrl.dismiss([formDictionary, this.vaccineSelected, this.vaccineOtherName, this.selectedRoute]);
        } else {
          this.modalCtrl.dismiss([formDictionary, this.vaccineSelected, null, this.selectedRoute]);
        }
      } else {
        /*
        * Else, the user navigated to the Health page normally via Actions page
        * Call data service to insert Health event into database
        */
        this.dataService.addVaccinationEvent(this.actionTypeID, action,
          this.itemType, this.vaccineIDs, this.selectedAmount, formDictionary)
          .then((response: any) => {
            /*
            * If there is a response (successful), then reset the list
            * Display message to user and navigate back to Actions page
            */
            if (response) {
              this.actionsService.actionToast('vaccination', this.itemType);
              this.router.navigateByUrl('/tabs/home');
            } else {
              // Else, there is not a response, display error message to user
              this.actionsService.actionError('Actions Vaccination Event Response Error');
            }
          })
          .catch((error: any) => {
            // Else, there is an unexpected error, display error message to user
            this.actionsService.actionError(error);
          });
      }
    }
  }

  // Set vaccine
  setVaccine(vaccine: any) {
    if (vaccine) {
      this.vaccineSelected = vaccine;
      this.isVaccineAvailable = false;

      // Patch the form value for `vaccine`
      this.vaccinationForm.patchValue({
        vaccine: this.vaccineSelected
      });

      if (vaccine === 'Other' && !this.isVaccineAvailable) {
        this.isVaccineOther = true;
      } else {
        this.isVaccineOther = false;
        this.vaccineOtherName = null;
      }
    }
  }

  // Grab vaccines data from local JSON file
  initializeItems() {
    return this.httpClient.get('/assets/data/vaccines.json');
  }

  // When the user begins searching (typing) a vaccine, filter results from `vaccines.json`
  getVaccines(event: any) {
    /*
    * If vaccines data has already been called and set into variable
        then only worry about filtering results based on user input
    */
    if (this.vaccines) {
      // Set `inputValue` to the value of the search bar
      const inputValue = event.target.value;

      // If `inputValue` is an empty string then don't filter the items
      if (inputValue && inputValue.trim() !== '') {
        this.isVaccineAvailable = true;
        this.filteredVaccines = this.vaccines.filter((item: any) =>

          /*
          * Return any occurrence of the inputValue in the string
          * i.e., `sp` would show a result that includes: `Anthrax Spore Vaccine`
          */
          (item.toLowerCase().indexOf(inputValue.toLowerCase()) > -1)
        );

      } else {
        /*
        * Else, the `inputValue` is empty, either there is no input or the user cleared existing input
        * Reset variables
        */
        this.isVaccineAvailable = false;
        this.isVaccineOther = false;
        this.vaccineSelected = null;
      }
    } else {
      /*
      * Else, no vaccine data has been called
      * Get data from local `vaccines.json` file
      * NOTE: This should be done on initial page load, however, if there is a problem
          attempt to retrieve the list of vaccine again
      */
      this.initializeItems().subscribe(resp => {
        if (resp) {
          this.vaccines = resp;
          this.getVaccines(event);
        } else {
          this.isVaccineAvailable = false;
        }
      });
    }
  }

  /*
  * If the user decides to cancel applying actions to their selected items
  * Navigate back to the main `Home` page
  */
  cancelActions() {
    this.router.navigateByUrl('/tabs/home');
  }

  // Reset form values
  resetForm() {
    this.vaccinationForm.patchValue({
      date: '',
      volume: '',
      units: '',
      vaccine: '',
      vaccineOther: '',
      route: '',
      serialNumber: '',
      expirationDate: ''
    });
  }

  ionViewDidEnter() {
    this.resetForm();
    if (!this.isModal) {
      /*
      * If there are selected values (animals or groups from the user)
      * Iterate through the selections and append the associated ID
          to the list of `vaccineIDs` used for writing records to database
      */
      if (this.selected) {
        for (const s of this.selected) {
          this.vaccineIDs.push(s.id);
        }
      } else {
        this.router.navigateByUrl('/tabs/home');
      }
    } else {
      console.log('Not a modal');
    }
  }

  ngOnInit() {
    // On initial page load, grab the list of vaccines from local device (JSON file)
    this.initializeItems().subscribe(resp => {
      this.dataService.presentLoading();
      if (resp) {
        this.vaccines = resp;
      }
    });
  }

}
