import { Injectable } from '@angular/core';
import { Router } from '@angular/router';
import { HttpClient } from '@angular/common/http';
import { AlertController } from '@ionic/angular';
import { Preferences } from '@capacitor/preferences';

@Injectable({
  providedIn: 'root'
})
export class AccountService {

  constructor(
    private router: Router,
    private httpClient: HttpClient,
    private alertCtrl: AlertController
  ) { }

  /**
   * Logout the user, clear their extID from device preferences, and navigate to the login page.
   */
  async logout() {
    await Preferences.remove({ key: 'extID' });
    this.router.navigateByUrl('/login', { replaceUrl: true });
  }

  /**
   * Display an alert with a message related to a login error.
   * @param error - The error message to display in the alert.
   */
  async loginAlert(error: string) {
    const alert = await this.alertCtrl.create({
      header: 'Login Error',
      message: error,
      buttons: ['OK']
    });

    await alert.present();
  }

  /**
   * Attempt to log the user in using their email and password.
   * @param email - The user's email address.
   * @param password - The user's password.
   */
  login(email: string, password: any) {
    // Check if both email and password are provided by the user
    if (email && password) {
      // Create a credentials object with the email and password
      const credentials = {
        email,
        password
      };

      // Make an HTTP POST request to the server with the provided credentials
      this.httpClient.post('https://calfstewardship.extension.msstate.edu/authenticate/login.php', credentials)
        .subscribe({
          // Handle successful response from the server
          next: async (response: any) => {
            // Check if the response is not empty
            if (response) {
              // If the response includes 'ext-', the user provided valid credentials
              if (response.includes('ext-')) {
                // Extract extID from response and trim any whitespace
                const extID = response.trim();

                // If extID has a value, store it in local device preferences and navigate to the Home page
                if (extID) {
                  await Preferences.set({
                    key: 'extID',
                    value: extID
                  });

                  this.router.navigateByUrl('/tabs/home');
                } else {
                  // If extID is empty, display an error message related to setting the username
                  this.loginAlert('Error occurred setting username. Please try again.');
                }
              } else {
                // If the response doesn't include 'ext-', the username was not found, and an alert is displayed
                this.loginAlert(response);
              }
            } else {
              // If there's no response from the server, display an alert
              this.loginAlert('No response. Please try again.');
            }
          },
          // Handle errors that occur during the HTTP request
          error: (error: any) => {
            // Log the error to the console
            console.log(`Login error: ${JSON.stringify(error)}`);

            // Display an alert with a generic error message
            this.loginAlert('Unexpected login error occurred. Please try again.');
          }
        });
    } else {
      // If email or password is not provided, display an alert asking the user to try again
      this.loginAlert('No credentials. Please try again.');
    }
  }

  // Check to see if the user is already logged in
  async checkLogin(page: string) {
    // Attempt to retrieve the extID value from local device preferences
    const { value } = await Preferences.get({ key: 'extID' });

    // If extID value exists, the user is already logged in
    if (value) {
      // If the user is logging in, direct them to the Home page
      if (page === 'login') {
        this.router.navigateByUrl('/tabs/home');
      } else if (page === 'home' || page === 'data') {
        // If the user is already logged in and we are checking if they should stay logged in,
        // this would occur on either the "Home" page or during a connection attempt on the "Data" page
        // If we are coming from either of these pages, return the value of the extID as a confirmation of their logged in status
        return value;
      } else {
        // Otherwise, in the case we are unexpectedly coming from another page, default to logout
        this.logout();
      }
    } else {
      // If there is no extID value, navigate the user to the Login page
      this.router.navigateByUrl('/login');
    }

    return null;
  }

}  