import { Injectable } from '@angular/core';
import { DataService } from './data.service';
import { AlertController } from '@ionic/angular';
import { Router } from '@angular/router';
import { Toast } from '@capacitor/toast';

@Injectable({
  providedIn: 'root'
})
export class AddService {

  constructor(
    private dataService: DataService,
    private alertCtrl: AlertController,
    private router: Router
  ) { }

  async showToast(message: string) {
    await Toast.show({
      duration: 'short',
      position: 'top',
      text: message
    });
  }

  // Display alert (with passed title and text) to user when necessary (i.e. an error occurs)
  async displayAlert(title: string, text: string) {
    const alert = await this.alertCtrl.create({
      header: title,
      message: text,
      buttons: [
        {
          text: 'OK'
        }
      ],
      backdropDismiss: false
    });
    await alert.present();
  }

  // Create multiple animals with defined prefix and sexes (bulls, steers, cows)
  createMultiple(groupID: number, groupName: string, additionType: number, prefix: string,
    bulls: number, steers: number, cows: number, birthDate: string, purchaseDate: string) {

    console.log(`createMultiple groupID ${groupID}`);
    console.log(`createMultiple birthDate ${birthDate}`);
    console.log(`createMultiple type birthDate ${typeof (birthDate)}`);
    console.log(`createMultiple purchaseDate ${purchaseDate}`);
    console.log(`createMultiple type purchaseDate ${typeof (purchaseDate)}`);

    this.dataService.createAnimals(groupID, additionType, prefix, bulls, steers, cows, birthDate, purchaseDate)
      .then((response: any) => {
        console.log(`createMultiple() response: ${response}`);
        if (response) {
          this.submitAlert('animal', groupID, groupName, response, 'createMultiple');
        } else {
          this.displayAlert('No Response: Create Multiple - Creating Animals',
            'An error occurred while trying to create multiple animals. Check your network connection and try again.');
        }
      })
      .catch((error) => {
        console.log(`createMultiple() error: ${error}`);
        this.displayAlert('Error: Create Multiple - Creating Animals',
          'An error occurred while trying to create multiple animals. Check your network connection and try again.');
      });
  }

  // Check response from user submission to server
  async submitAlert(category: string, groupID: any, groupName: any, response: any, func: string) {
    /*
    * If server response was not "success" (string), then an error occurred inserting new records into database
    * Display alert to the user
    */
    console.log(`submitAlert() response: ${response}`);
    if (response !== 'success') {
      this.displayAlert('Error Alert',
        'An error occurred while trying to add animals. Check your network connection and try again.');
    } else {
      /*
      * Check if the current category is `animal` or `group`
      * This will help determine where to redirect the view
      */
      if (category === 'animal') {
        // Set navigation based on function call
        if (func === 'createMultiple') {
          this.router.navigateByUrl('/tabs/home/manage');
        } else {
          this.router.navigateByUrl(`/tabs/home/manage/group-animals/${groupID}/${groupName}/0`);
        }
      } else if (category === 'group') {
        /*
        * Otherwise, data was successfully inserted
        * Navigate back to the manage page (list of groups)
        */
        this.router.navigateByUrl('/tabs/home/manage');
      }
    }
  }

  addMultiple(category: any, groupID: any, groupName: any, formDataList: any) {
    /*
    * Iterate through the multiple forms and send (upload) one at a time to the server
    * If a form fails, display alert, otherwise continue
    * Show submission alert if all forms were successfully submitted and data inserted to database
    */
    for (const form of formDataList) {
      if (category === 'animal') {
        this.dataService.add('addAnimals', form)
          .then((response: any) => {
            console.log(`addMultiple() addAnimals response: ${response}`);

            // If there was not a server response, display error alert
            if (!response) {
              this.displayAlert('Error Adding Animals',
                'An error occurred while trying to add multiple animals. Check your network connection and try again.');
            } else {
              // Otherwise, display submission alert
              this.submitAlert(category, groupID, groupName, 'success', 'addMultipleAnimal');
            }
          })
          .catch((error) => {
            console.log(`addMultiple() error: ${error}`);
            this.displayAlert('Error Adding Animals',
              'An error occurred while trying to add multiple animals. Check your network connection and try again.');
          });

      } else if (category === 'group') {
        this.dataService.add('addGroups', form)
          .then((response: any) => {
            console.log(`addMultiple() addGroups response: ${response}`);

            // If there was not a server response, display error alert
            if (!response) {
              this.displayAlert('Error Adding Groups',
                'An error occurred while trying to add multiple groups. Check your network connection and try again.');
            } else {
              // Otherwise, display submission alert
              this.submitAlert(category, null, null, 'success', 'addMultipleGroup');
            }
          })
          .catch((error) => {
            console.log(`addMultiple() addGroups error: ${error}`);
            this.displayAlert('Error Adding Groups',
              'An error occurred while trying to add multiple groups. Check your network connection and try again.');
          });
      }
    }
  }

  // Add an individual item (animal or group)
  addIndividual(category: any, groupID: any, groupName: any, formData: any) {
    if (category === 'animal') {
      // Send group form data function that uploads to database via PHP script
      this.dataService.add('addAnimals', formData)
        .then((response: any) => {
          console.log(`addIndividual() addAnimals response: ${response}`);

          /*
          * If response was received from server, call submitAlert()
          * Pass category (e.g., `animal`), groupID (e.g., 33 [group animal is in]), and server response
          */
          if (response) {
            this.submitAlert(category, groupID, groupName, response, 'addIndividualAnimal');
          } else {
            this.displayAlert('Error Adding Animal',
              'An error occurred while trying to add animal. Check your network connection and try again.');
          }
        })
        .catch((error) => {
          console.log(`addIndividual() addAnimals error: ${error}`);
          this.displayAlert('Error Adding Animal',
            'An error occurred while trying to add animal. Check your network connection and try again.');
        });
    } else if (category === 'group') {
      // Send group form data function that uploads to database via php
      this.dataService.add('addGroups', formData)
        .then((response: any) => {
          console.log(`addIndividual() addGroups response: ${response}`);
          if (response) {
            // If a response was received from the server, display alert
            this.submitAlert(category, null, null, response, 'addIndividualGroup');
          } else {
            this.displayAlert('Error Adding Group',
              'An error occurred while trying to add group. Check your network connection and try again.');
          }
        })
        .catch((error) => {
          console.log(`addIndividual() addAnimals error: ${error}`);
          this.displayAlert('Error Adding Group',
            'An error occurred while trying to add group. Check your network connection and try again.');
        });
    }
  }

  /*
  * Prompt the user if they want to cancel their adding of items
  * Let the user know the current data (individual or multiple) will be lost if they exit the current page
  */
  async cancelAdd(viewType: any, groupID: any, groupName: any) {
    const alert = await this.alertCtrl.create({
      header: 'Exit Add',
      message: 'Are you sure you want to leave this page? ' +
        'All data you have entered will be lost and you will be returned to the Home page!',
      buttons: [
        {
          text: 'Leave Page',
          cssClass: 'alertRed',
          handler: () => {
            // Set navigation based on the current page view type
            if (viewType === 'group') {
              this.router.navigateByUrl('/tabs/home/manage');
            } else if (viewType === 'animal' && groupID && groupName) {
              this.router.navigateByUrl(`/tabs/home/manage/group-animals/${groupID}/${groupName}/0`);
            } else {
              this.router.navigateByUrl('/tabs/home/manage');
            }
          }
        },
        {
          text: 'Dismiss',
          role: 'cancel',
          cssClass: 'alertGreen',
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
  }

  /*
  * If the user selected to set a prefix
  * Prompt user to enter the prefix string and an optional starting prefix number
  */
  async definePrefix(): Promise<any> {
    let resolveFunction: (data: any) => void;

    const promise = new Promise<any>(resolve => {
      resolveFunction = resolve;
    });

    const alert = await this.alertCtrl.create({
      header: 'Define Name Prefix',
      message: 'Please enter an name prefix for all items about to added. This can be changed later.',
      inputs: [
        {
          name: 'prefix',
          type: 'text',
          placeholder: 'enter name prefix'
        },
        {
          name: 'startingNumber',
          type: 'text',
          placeholder: 'enter starting number (optional)'
        }
      ],
      buttons: [
        {
          text: 'Cancel',
          role: 'cancel',
          cssClass: 'alertRed'
        },
        {
          text: 'Set Prefix',
          cssClass: 'alertGreen',
          handler: (data) => {
            // Return the input (prefix and optional starting prefix number)
            resolveFunction(data);
          }
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
    return promise;
  }

  // Prompt the user to define an ID prefix (optional) for automating naming (ID) when adding multiple items
  async askPrefix(): Promise<any> {
    let resolveFunction: (result: any) => void;

    // Set a promise inside of the alert input to return the input text from the user
    const promise = new Promise<any>(resolve => {
      resolveFunction = resolve;
    });

    const alert = await this.alertCtrl.create({
      header: 'Define Prefix',
      message: 'Would you like to assign a prefix to the items about to be added? ' +
        'This will make naming automated so each item will not have to be identified individually. You can always change the name.',
      buttons: [
        {
          text: 'No',
          role: 'cancel',
          cssClass: 'alertRed',
        },
        {
          text: 'Yes',
          cssClass: 'alertGreen',
          handler: () => {
            // Return the prefix result from the user input
            resolveFunction(this.definePrefix());
          }
        }
      ],
      backdropDismiss: false
    });

    await alert.present();
    return promise;
  }
}
