import { Injectable } from '@angular/core';
import { HttpClient } from '@angular/common/http';
import { AccountService } from './account.service';
import { LoadingController } from '@ionic/angular';
import { firstValueFrom } from 'rxjs';

@Injectable({
  providedIn: 'root'
})
export class DataService {

  private apiURL = 'https://calfstewardship.extension.msstate.edu/app/';

  constructor(
    private accountService: AccountService,
    private httpClient: HttpClient,
    private loadingCtrl: LoadingController
  ) { }

  // Present loading spinner on screen while we retrieve data
  async presentLoading() {
    const loading = await this.loadingCtrl.create({
      message: 'Loading...',
      spinner: 'crescent',
      duration: 500
    });

    await loading.present();
  }

  // Method to generate a report given a reportType, data, selected items, and itemType
  async generateReport(reportType: string, data: any, selected: any, itemType: string) {
    return new Promise(async (resolve, reject) => {

      console.log('generateReport()');
      console.log('generateReport() reportType:', reportType);
      console.log('generateReport() data:', data);
      console.log('generateReport() selected:', selected);

      // Checking whether the user is logged in, userID will be null if not
      const userID = await this.accountService.checkLogin('data');

      // If userID is null, reject the Promise and return
      if (!userID) {
        console.log('generateReport Reports Reject 2');
        reject(null);
        return;
      }

      // If there is neither data nor selected, log an error message and return
      if (!data && !selected) {
        console.log('No data for generating report');
        return;
      }

      // Variable to store data to be passed for report generation
      let passedData = null;

      // Array of special report types that require different data structure
      const specialReportTypes = ['healthEvent', 'customProcessing', 'birth', 'pcr', 'ccr', 'cpr', 'growth', 'air', 'bior', 'nsbr'];

      // If reportType is in the specialReportTypes array, process data and selected items accordingly
      if (specialReportTypes.includes(reportType)) {
        console.log('passedData object');

        // If itemType is 'all', set targetField to 'selected', else keep itemType as it is
        const targetField = itemType === 'all' ? 'selected' : itemType;

        // Construct the passedData object with data and selected items
        passedData = {
          data,
          [targetField]: selected
        };
      } else {
        // If reportType is not in specialReportTypes, either data or selected will be passed as is
        passedData = data || selected;
      }

      // If there is no reportType or no data to pass, log an error message and return
      if (!reportType || !passedData) {
        console.log('No report type or passed data for generating report!');
        return;
      }

      console.log('generateReport() reportType', reportType);
      console.log('generateReport() passedData', passedData);

      // Making a call to the backend API to generate the report
      // Data is passed as an array which includes reportType, userID, a boolean and the passedData object
      this.httpClient.post(`${this.apiURL}reports.php`, [reportType, userID, true, passedData])
        .subscribe(response => {
          // If response exists, resolve the Promise with the response, else reject it with null
          if (response) {
            resolve(response);
          } else {
            console.log('generateReport Reports Response Reject 1:', response);
            reject(null);
          }
        });

    });
  }


  async addBreedingEvent(actionTypeID: number, action: string, itemType: string,
    cowIDs: any, selectedAmount: string, formDictionary: any) {
    return new Promise(async (resolve, reject) => {
      // Check to see that the user has logged in
      const userID = await this.accountService.checkLogin('data');

      // If there is a value, `extID`, then proceed with making API call using appropriate data
      if (userID && formDictionary) {

        const data = {
          actionTypeID,
          action,
          itemType,
          cowIDs,
          selectedAmount,
          userID,
          loggedIn: true
        };

        /*
        * Pass a list of the `data` object and `formDictionary` object to the server script
        * NOTE: Had difficulty handling the `data` object with a `formData` object,
            so a dictionary was created and used instead
        */
        this.httpClient.post(`${this.apiURL}actionsBreedingEvent.php`, [data, formDictionary])
          .subscribe(response => {
            console.log('actionsBreedingEvent response:', response);
            if (response && response === 'success') {
              resolve(response);
            } else {
              console.log('actionsBreedingEvent Response Reject:', response);
              reject('Actions Breeding Event Response Error');
            }
          });
      } else {
        reject('No Actions Health data available!');
      }
    });
  }

  async addCustomProcessingFormEvent(actionTypeID: number, action: string,
    selectedAmount: string, selectedAnimals: any, additionalOptions: any, formDataList: any) {
    return new Promise(async (resolve, reject) => {

      // Check to see that the user has logged in
      const userID = await this.accountService.checkLogin('data');

      if (userID && formDataList.length > 0 && action) {
        /*
        * Iterate through list of form data and send (upload) one at a time to the server
        * If a form fails, display alert, otherwise continue
        * Show submission alert if all forms were successfully submitted and data inserted to database
        */
        const data = {
          actionTypeID,
          action,
          selectedAmount,
          userID,
          loggedIn: true
        };

        console.log('data:', data);
        console.log('formDataList:', formDataList);
        console.log('selectedAnimals:', selectedAnimals);
        console.log('additionalOptions:', additionalOptions);

        /*
        * Pass a list of the `data` object and `formDictionary` object to the server script
        * NOTE: Had difficulty handling the `data` object with a `formData` object,
            so a dictionary was created and used instead
        */
        this.httpClient.post(`${this.apiURL}actionsCustomProcessingFormEvent.php`,
          [data, formDataList, selectedAnimals, additionalOptions])
          .subscribe(response => {
            console.log('actionsCustomProcessingFormEvent response:', response);
            if (response && response === 'success') {
              resolve(response);
            } else {
              console.log('actionsCustomProcessingFormEvent Response Reject:', response);
              reject('Actions Custom Processing Form Event Response Error');
            }
          });
      } else {
        reject('No Actions Custom Processing Form Event data available!');
      }
    });
  }

  async addProcessingEvent(actionTypeID: number, action: string, itemType: string,
    processingIDs: any, selectedAmount: string, formDictionary: any) {
    return new Promise(async (resolve, reject) => {
      // Check to see that the user has logged in
      const userID = await this.accountService.checkLogin('data');

      // If there is a value, `extID`, then proceed with making API call using appropriate data
      if (userID && formDictionary) {

        const data = {
          actionTypeID,
          action,
          itemType,
          processingIDs,
          selectedAmount,
          userID,
          loggedIn: true
        };

        /*
        * Pass a list of the `data` object and `formDictionary` object to the server script
        * NOTE: Had difficulty handling the `data` object with a `formData` object,
            so a dictionary was created and used instead
        */
        this.httpClient.post(`${this.apiURL}actionsProcessingEvent.php`, [data, formDictionary])
          .subscribe(response => {
            console.log('actionsProcessingEvent response:', response);
            if (response && response === 'success') {
              resolve(response);
            } else {
              console.log('actionsProcessingEvent Response Reject:', response);
              reject(null);
            }
          });
      } else {
        reject(null);
      }
    });
  }

  async addVaccinationEvent(actionTypeID: number, action: string, itemType: string,
    vaccineIDs: any, selectedAmount: string, formDictionary: any) {
    return new Promise(async (resolve, reject) => {
      // Check to see that the user has logged in
      const userID = await this.accountService.checkLogin('data');

      // If there is a value, `extID`, then proceed with making API call using appropriate data
      if (userID && formDictionary) {

        const data = {
          actionTypeID,
          action,
          itemType,
          vaccineIDs,
          selectedAmount,
          userID,
          loggedIn: true
        };

        /*
        * Pass a list of the `data` object and `formDictionary` object to the server script
        * NOTE: Had difficulty handling the `data` object with a `formData` object,
            so a dictionary was created and used instead
        */
        this.httpClient.post(`${this.apiURL}actionsVaccinationEvent.php`, [data, formDictionary])
          .subscribe(response => {
            console.log('actionsVaccinationEvent response:', response);
            if (response && response === 'success') {
              resolve(response);
            } else {
              reject('Actions Vaccination Event Response Error');
            }
          });
      } else {
        reject('No Actions Vaccination Event data available!');
      }
    });
  }

  async addHealthEvent(actionTypeID: number, action: string, itemType: string,
    healthIDs: any, selectedAmount: string, formDictionary: any) {
    return new Promise(async (resolve, reject) => {
      // Check to see that the user has logged in
      const userID = await this.accountService.checkLogin('data');

      // If there is a value, `extID`, then proceed with making API call using appropriate data
      if (userID && formDictionary) {

        const data = {
          actionTypeID,
          action,
          itemType,
          healthIDs,
          selectedAmount,
          userID,
          loggedIn: true
        };

        console.log(data);

        /*
        * Pass a list of the `data` object and `formDictionary` object to the server script
        * NOTE: Had difficulty handling the `data` object with a `formData` object,
            so a dictionary was created and used instead
        */
        this.httpClient.post(`${this.apiURL}actionsHealthEvent.php`, [data, formDictionary])
          .subscribe(response => {
            console.log('actionsHealthEvent response:', response);
            if (response && response === 'success') {
              resolve(response);
            } else {
              reject('Actions Health Event Response Error');
            }
          });
      } else {
        reject('No Actions Health data available!');
      }
    });
  }

  // Add existing animal to another group (already exists in one)
  addExistingAnimalsToGroup(groupID: number, animalIDs: any) {
    return new Promise(async (resolve, reject) => {

      // Check to see that the user has logged in
      const userID = await this.accountService.checkLogin('data');

      // If there is a value, `extID`, then proceed with making API call using appropriate data
      if (userID && groupID && animalIDs) {

        // Declare `data` object with `userID` and `loggedIn` values
        const data = {
          groupID,
          animalIDs,
          userID,
          loggedIn: true
        };

        this.httpClient.post(`${this.apiURL}addExistingAnimalsToGroup.php`, data)
          .subscribe(response => {
            console.log('addExistingAnimalsToGroup response:', response);
            if (response && response === 'success') {
              resolve(response);
            } else {
              console.log('addExistingAnimalsToGroup Response Reject:', response);
              reject(null);
            }
          });
      } else {
        reject(null);
      }
    });
  }

  // Edit specific group for current user
  editGroup(formData: any) {
    return new Promise(async (resolve, reject) => {

      // Check to see that the user has logged in
      const userID = await this.accountService.checkLogin('data');

      // If there is a value, `extID`, then proceed with making API call using appropriate data
      if (userID && formData) {

        // Append values to passed formData
        formData.append('userID', userID);
        formData.append('loggedIn', true);

        this.httpClient.post(`${this.apiURL}editGroup.php`, formData)
          .subscribe(response => {
            console.log('editGroup response:', response);
            if (response) {
              resolve(response);
            } else {
              console.log('editGroup Response Reject:', response);
              reject(null);
            }
          });
      } else {
        reject(null);
      }
    });
  }


  // Edit specific animal for current user
  editAnimals(formData: any) {
    return new Promise(async (resolve, reject) => {

      // Check to see that the user has logged in
      const userID = await this.accountService.checkLogin('data');

      // If there is a value, `extID`, then proceed with making API call using appropriate data
      if (userID && formData) {

        // Append values to passed formData
        formData.append('userID', userID);
        formData.append('loggedIn', true);

        this.httpClient.post(`${this.apiURL}editAnimals.php`, formData)
          .subscribe(response => {
            console.log('editAnimals response:', response);
            if (response) {
              resolve(response);
            } else {
              reject(null);
            }
          });
      } else {
        reject(null);
      }
    });
  }


  category(name: any, category: any, categoryName: any, newCategory: any) {
    return new Promise(async (resolve, reject) => {

      // Check to see that the user has logged in
      const userID = await this.accountService.checkLogin('data');

      // Initialize `data` to empty object; values filled based on `name`
      let data = {};

      // If there is a value, `extID`, then proceed with making API call using appropriate data
      if (userID) {

        /*
        * Determine the appropriate PHP script to use related to category functionality
        * Set appropriate data
        */
        if (name === 'addGroupCategory') {
          data = {
            categoryName,
            userID,
            loggedIn: true
          };
        }
        else if (name === 'checkGroupCategory') {
          data = {
            categoryName: category.category_name,
            userID,
            loggedIn: true
          };
        } else if (name === 'deleteGroupCategory') {
          data = {
            categoryID: category.id,
            userID,
            loggedIn: true
          };
        } else if (name === 'updateGroupCategory') {
          data = {
            oldCategoryID: category.id,
            newCategoryID: newCategory.id,
            userID,
            loggedIn: true
          };
        } else if (name === 'renameGroupCategory') {
          data = {
            rename: categoryName,
            categoryID: category.id,
            userID,
            loggedIn: true
          };
        }

        if (data && name) {
          this.httpClient.post(`${this.apiURL}${name}.php`, data)
            .subscribe(response => {
              if (name === 'checkGroupCategory' && +response >= 0) {
                resolve(response);
              }
              else if ((name === 'renameGroupCategory' || name === 'deleteGroupCategory') && +response === 1) {
                resolve(response);
              }
              else if (response === 'success') {
                resolve(response);
              } else {
                reject(null);
              }
            });
        } else {
          reject(null);
        }
      } else {
        reject(null);
      }
    });
  }

  // Create specific number of animals per sex with a defined prefix string for the initial ID
  async createAnimals(
    groupID: number,
    additionType: number,
    prefix: string,
    bulls: number,
    steers: number,
    cows: number,
    birthDate: string,
    purchaseDate: string
  ): Promise<string | null> {
    console.log(`createAnimals() groupID ${groupID}`);
    console.log(`createAnimals() birthDate ${birthDate}`);
    console.log(`createAnimals() purchaseDate ${purchaseDate}`);

    // Check to see that the user has logged in
    const userID = await this.accountService.checkLogin('data');

    const createString = `${groupID},${additionType},${prefix},${bulls},${steers},${cows},${birthDate},${purchaseDate},${userID}`;

    console.log(`createAnimals() createString ${createString}`);

    if (userID) {
      try {
        const response$ = this.httpClient.post<string>(
          `${this.apiURL}createAnimals.php`,
          createString
        );
        const response = await firstValueFrom(response$);

        console.log('createAnimals() response:', response);
        if (response === 'success') {
          return response;
        } else {
          return null;
        }
      } catch (error) {
        console.error('createAnimals() error:', error);
        return null;
      }
    } else {
      return null;
    }
  }

  // Make call to API to add data to database
  add(filename: string, formData: any) {
    return new Promise(async (resolve, reject) => {

      // Check to see that the user has logged in
      const userID = await this.accountService.checkLogin('data');

      // If there is a value, `extID`, then proceed with making API call using appropriate data
      if (userID && filename && formData) {

        // Append values to passed formData
        formData.append('userID', userID);
        formData.append('loggedIn', true);

        // Make call to API, passing appropriate data
        this.httpClient.post(`${this.apiURL}${filename}.php`, formData)
          .subscribe(response => {
            if (response) {
              resolve(response);
            } else {
              reject(null);
            }
          });
      } else {
        reject(null);
      }
    });
  }

  async get(filename: string, groupID: any): Promise<any> {
    console.log(`get() filename: ${filename}`);
    // Check to see that the user has logged in
    const userID = await this.accountService.checkLogin('data');

    // If there is a value, `userID`, then proceed with making API call using appropriate data
    if (userID && filename) {
      console.log(`get() userID: ${userID}`);
      // Declare `data` object with `userID`, `loggedIn`, and `groupID` values
      const data = {
        userID,
        loggedIn: true,
        groupID
      };

      try {
        // Make call to API, passing appropriate data
        const response: any = await firstValueFrom(this.httpClient.post<any>(`${this.apiURL}${filename}.php`, data));
        // console.log(`get() response: ${JSON.stringify(response)}`)

        // TODO: Handle this better
        // If the response is a string, then that means we have received a returned error message (string, json encoded)
        // If the response is not a string, then proceed with processing the data and resolve (return) to the appropriate page
        if (typeof response !== 'string') {
          return response;
        } else {
          return null;
        }
      } catch (error) {
        console.error(`get() ${filename} error: ${JSON.stringify(error)}`);
        return null;
      }
    } else {
      return null;
    }
  }


}
