import { Injectable } from '@angular/core';
import { Share } from '@capacitor/share';
import { Filesystem, Directory, Encoding } from '@capacitor/filesystem';

@Injectable({
  providedIn: 'root',
})
export class ShareService {
  constructor() { }

  private getCurrentDateTime(): string {
    const now = new Date();
    return `${now.getFullYear()}-${now.getMonth() + 1}-${now.getDate()}_${now.getHours()}-${now.getMinutes()}-${now.getSeconds()}`;
  }

  private async jsonToCSV(jsonData: any[], excludeColumns: string[] = []): Promise<string> {
    const replacer = (key: string, value: any) => (value === null ? '' : value);
    const header = Object.keys(jsonData[0]).filter(column => !excludeColumns.includes(column));
    const csv = [
      header.join(','),
      ...jsonData.map(row =>
        header
          .map(fieldName => JSON.stringify(row[fieldName], replacer))
          .join(',')
      ),
    ].join('\r\n');

    return csv;
  }

  private async saveCSVFile(fileName: string, csvData: string): Promise<string> {
    const result = await Filesystem.writeFile({
      path: fileName,
      data: csvData,
      directory: Directory.Documents,
      encoding: Encoding.UTF8,
    });

    return result.uri;
  }

  async generateAndShareCSV(reportName: string, data: any[], excludeColumns: string[] = []): Promise<void> {
    try {
      // Check for permissions
      let permissionStatus = await Filesystem.checkPermissions();
      if (permissionStatus.publicStorage !== 'granted') {
        // Request for permissions if not granted
        permissionStatus = await Filesystem.requestPermissions();
        if (permissionStatus.publicStorage !== 'granted') {
          throw new Error('CALF Manager does not have permission to write files.');
        }
      }

      const dateTime = this.getCurrentDateTime();

      const csvData = await this.jsonToCSV(data, excludeColumns);
      const fileName = `${reportName}_report_${dateTime}.csv`;
      const fileUri = await this.saveCSVFile(fileName, csvData);

      await Share.share({
        title: `CALF Manager ${reportName} Report ${dateTime}`,
        url: fileUri,
      });

    } catch (error) {
      console.error(`Error creating CSV and sharing for CALF Manager ${reportName} Report:`, error);
    }
  }
}
