<?php
require('accessControl.php');

/*
BY:				krn
FILENAME:		actionsCustomProcessingFormEvent.php
DESCRIPTION:	This PHP script manages custom processing form events by defining arrays for custom, health,
				and vaccination objects. It includes functions for updating custom events, checking IDs,
				creating vaccination and treatment health events, creating custom events, and retrieving the
				custom action ID. Upon receiving a POST request with specific data, the script
				validates user authentication and processes the custom form data. It inserts custom events
				into the database and handles additional options like treatments and vaccinations.
				The script updates the associated animal IDs in custom event records, and returns a
				JSON response indicating success or failure before closing the database connection.
*/

// define local time zone and format
date_default_timezone_set('America/Chicago');

// Define array to hold custom event IDs for all inserted records into event_custom table
$allCustomObjects = [];

// Function to write a new record to event_health_mapping table
function insertHealthMappingRecord($mysqli, $healthID, $eventID)
{
	$updateHealthMappingQuery = $mysqli->prepare('INSERT INTO `event_health_mapping` (`health_id`, `event_id`, `event_datetime`) VALUES (?, ?, ?)');
	$eventDateTime = date('Y-m-d H:i:s');
	if ($updateHealthMappingQuery) {
		$updateHealthMappingQuery->bind_param('iis', $healthID, $eventID, $eventDateTime);
		$updateHealthMappingQuery->execute();
		$updateHealthMappingQuery->close();
	} else {
		echo json_encode($mysqli->error);
		exit(1);
	}
}

// Function to write a new record to event_vaccination_mapping table
function insertVaccinationMappingRecord($mysqli, $vaccinationID, $eventID)
{
	$updateVaccinationMappingQuery = $mysqli->prepare('INSERT INTO `event_vaccination_mapping` (`vaccination_id`, `event_id`, `event_datetime`) VALUES (?, ?, ?)');
	$eventDateTime = date('Y-m-d H:i:s');
	if ($updateVaccinationMappingQuery) {
		$updateVaccinationMappingQuery->bind_param('iis', $vaccinationID, $eventID, $eventDateTime);
		$updateVaccinationMappingQuery->execute();
		$updateVaccinationMappingQuery->close();
	} else {
		echo json_encode($mysqli->error);
		exit(1);
	}
}

// Function to handle Vaccination events
function vaccinationEvent($mysqli, $id, $actionTypeID, $actionID, $itemType, $userID, $vaccinationData)
{
	if ($itemType == 'animals') {
		$insertVaccinationEventQuery = $mysqli->prepare('INSERT INTO `event_vaccination`
            (`action_type_id`, `action_id`, `animal_id`, `date`, `volume`, `units_id`, `vaccine`, `vaccine_other`,
            `roa_id`, `serial_number`, `expiration_date`, `user_id`, `event_datetime`)
            VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?)');
	} elseif ($itemType === 'groups') {
		$insertVaccinationEventQuery = $mysqli->prepare('INSERT INTO `event_vaccination`
            (`action_type_id`, `action_id`, `group_id`, `date`, `volume`, `units_id`, `vaccine`, `vaccine_other`,
            `roa_id`, `serial_number`, `expiration_date`, `user_id`, `event_datetime`)
            VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?)');
	} else {
		echo json_encode('Invalid Vaccination Item Type!');
		exit(1);
	}

	$eventDateTime = date('Y-m-d H:i:s');

	if (
		$insertVaccinationEventQuery &&
		$insertVaccinationEventQuery->bind_param(
			'iiisdississss',
			$actionTypeID, // action type ID
			$actionID, // action ID
			$id, // ID of item (animal or group)
			$vaccinationData[0], // date
			$vaccinationData[1], // volume (optional)
			$vaccinationData[2], // units
			$vaccinationData[3], // vaccine
			$vaccinationData[4], // vaccine other (if the user selects "Other" as vaccine and inputs another vaccine)
			$vaccinationData[5], // route of administration
			$vaccinationData[6], // serial number (optional)
			$vaccinationData[7], // expiration date (optional)
			$userID,
			$eventDateTime
		) &&
		$insertVaccinationEventQuery->execute()
	) {
		$insertVaccinationEventQuery->close();
		return $mysqli->insert_id;
	} else {
		echo json_encode($mysqli->error);
		exit(1);
	}
}

// Function to handle Health (Treatment) events
function treatmentHealthEvent($mysqli, $id, $actionTypeID, $actionID, $itemType, $userID, $healthData)
{
	if ($itemType == 'animals') {
		$insertHealthEventQuery = $mysqli->prepare('INSERT INTO `event_health`
            (`action_type_id`, `action_id`, `animal_id`, `date`, `volume`, `units_id`, `product`, `product_other`,
            `roa_id`, `reason`, `problem_bs`, `problem_bsc`, `user_id`, `event_datetime`)
            VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?)');
	} elseif ($itemType === 'groups') {
		$insertHealthEventQuery = $mysqli->prepare('INSERT INTO `event_health`
            (`action_type_id`, `action_id`, `group_id`, `date`, `volume`, `units_id`, `product`, `product_other`,
            `roa_id`, `reason`, `problem_bs`, `problem_bsc`, `user_id`, `event_datetime`)
            VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?)');
	} else {
		echo json_encode('Invalid Health (Treatment) Item Type!');
		exit(1);
	}

	$eventDateTime = date('Y-m-d H:i:s');

	if (
		$insertHealthEventQuery &&
		$insertHealthEventQuery->bind_param(
			'iiisdissiiiiss',
			$actionTypeID, // action type ID
			$actionID, // action ID
			$id, // ID of item (animal or group)
			$healthData[0], // date
			$healthData[1], // volume (optional)
			$healthData[2], // units
			$healthData[3], // product
			$healthData[4], // product other (if the user selects "Other" as product and inputs another product)
			$healthData[5], // route of administration
			$healthData[6], // reason
			$healthData[7], // problem body system
			$healthData[8], // problem body system condition
			$userID, // user ID
			$eventDateTime // database record date
		) &&
		$insertHealthEventQuery->execute()
	) {
		$insertHealthEventQuery->close();
		return $mysqli->insert_id;
	} else {
		echo json_encode($mysqli->error);
		exit(1);
	}
}


function customEvent($mysqli, $id, $actionTypeID, $actionID, $itemType, $userID, $customData)
{

	if ($itemType === 'animals') {
		$insertCustomEventQuery = $mysqli->prepare('INSERT INTO `event_custom`
        (`action_type_id`, `action_id`, `animal_id`, `weaning_weight`, `yearling_weight`, `mature_weight`, `hip_height`, `bcs`, `repro_tract_score`,
        `chute_docility_score`, `foot_angle_score`, `claw_set_score`, `teat_score`, `udder_score`, `bse_motility`, `bse_morphology`, `bse_acceptable`,
        `pregnancy_status`, `gestation_length`, `gl_units`, `castration_tech`, `castration_date`, `dehorning_tech`, `dehorning_date`, `spaying_tech`,
        `spaying_date`, `body_temperature`, `notes`, `user_id`, `event_datetime`)
        VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)');
	} elseif ($itemType === 'groups') {
		$insertCustomEventQuery = $mysqli->prepare('INSERT INTO `event_custom`
        (`action_type_id`, `action_id`, `group_id`, `weaning_weight`, `yearling_weight`, `mature_weight`, `hip_height`, `bcs`, `repro_tract_score`,
        `chute_docility_score`, `foot_angle_score`, `claw_set_score`, `teat_score`, `udder_score`, `bse_motility`, `bse_morphology`, `bse_acceptable`,
        `pregnancy_status`, `gestation_length`, `gl_units`, `castration_tech`, `castration_date`, `dehorning_tech`, `dehorning_date`, `spaying_tech`,
        `spaying_date`, `body_temperature`, `notes`, `user_id`, `event_datetime`)
        VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)');
	} else {
		echo json_encode('Invalid Custom Item Type!');
		exit(1);
	}

	$eventDateTime = date('Y-m-d H:i:s');

	if (
		$insertCustomEventQuery &&
		$insertCustomEventQuery->bind_param(
			'iiiddddiiiiiiiiiisdsisisisdsss',
			$actionTypeID,		// action type ID
			$actionID,			// action ID
			$id,				// ID of item (animal or group)
			$customData[0],		// weaning weight
			$customData[1],		// yearling weight
			$customData[2],		// mature weight
			$customData[3],		// hip height
			$customData[4],		// body condition score (bcs)
			$customData[5],		// reproductive tract score (cows only)
			$customData[6],		// chute docility score
			$customData[7],		// foot angle score
			$customData[8],		// claw set score
			$customData[9],		// teat score (cows only)
			$customData[10],	// udder score (cows only)
			$customData[11],	// bse motility (breeding soundness exam) (bulls only)
			$customData[12],	// bse morphology (breeding soundness exam) (bulls only)
			$customData[13],	// bse acceptable (breeding soundness exam) (bulls only)
			$customData[14],	// pregnancy status (cows only)
			$customData[15],	// pregnancy gestation length (cows only)
			$customData[16],	// pregnancy gestation length units (cows only)
			$customData[17],	// castration technique (bulls only)
			$customData[18],	// castration date (bulls only)
			$customData[19],	// dehorning technique (all)
			$customData[20],	// dehorning date (all)
			$customData[21],	// spaying technique (cows only)
			$customData[22],	// spaying date (cows only)
			$customData[23],	// body temperature
			$customData[24],	// notes
			$userID,			// user ID
			$eventDateTime		// database record date
		) &&
		$insertCustomEventQuery->execute()
	) {
		$insertCustomEventQuery->close();
		return $mysqli->insert_id;
	} else {
		echo json_encode($mysqli->error);
		exit(1);
	}
}

function getCustomActionID($mysqli, $actionTypeID, $action, $itemType, $selectedAmount)
{
	$selectActionIDQuery = $mysqli->prepare(
		'SELECT `id`
        FROM `actions`
        WHERE `action_type_id` = ?
        AND `action` = ?
        AND `item` = ?
        AND `amount` = ?'
	);

	$actionID = NULL;

	if (
		$selectActionIDQuery &&
		$selectActionIDQuery->bind_param(
			'isss',
			$actionTypeID,
			$action,
			$itemType,
			$selectedAmount
		) &&
		$selectActionIDQuery->execute()
	) {
		$selectActionIDQuery->store_result();
		$selectActionIDQuery->bind_result($actionID);
		$selectActionIDQuery->fetch();
		return $actionID;
	} else {
		echo json_encode($mysqli->error);
		exit(1);
	}
}


// make sure we have received a post request from app
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
	$data = file_get_contents('php://input');
	$request = json_decode($data);

	if (isset($request[0], $request[1])) {
		require('connection.php');

		/*
		$request[0] = {actionTypeID, action, itemType, customIDs, userID, loggedIn}
		array(5) {
			[0]=>
			object(stdClass)#1 (7) {
				["actionTypeID"]=>
				int(4)
				["action"]=>
				string(6) "custom"
				["itemType"]=>
				string(7) "animals"
				["customIDs"]=>
				array(1) {
				[0]=>
				int(878)
				}
				["selectedAmount"]=>
				string(6) "single"
				["userID"]=>
				string(8) "ext-ct38"
				["loggedIn"]=>
				bool(true)
			}
		$request[1] is an array of objects, each object representing a custom form fields/input
		  [1]=>
			array(1) {
				[0]=>
				object(stdClass)#2 (26) {
				["id"]=>
				int(878)
				["weaningWeight"]=>
				string(0) ""
				["yearlingWeight"]=>
				string(0) ""
				["matureWeight"]=>
				string(0) ""
				["hipHeight"]=>
				string(0) ""
				["bodyConditionScore"]=>
				string(0) ""
				["reproductiveTractScore"]=>
				string(0) ""
				["chuteDocilityScore"]=>
				string(0) ""
				["footAngleScore"]=>
				string(0) ""
				["clawSetScore"]=>
				string(0) ""
				["teatScore"]=>
				string(0) ""
				["udderScore"]=>
				string(0) ""
				["bseMotility"]=>
				string(0) ""
				["bseMorphology"]=>
				string(0) ""
				["bseAcceptableBreeder"]=>
				string(0) ""
				["pregnancyStatus"]=>
				string(0) ""
				["gestationLength"]=>
				string(0) ""
				["gestationLengthUnits"]=>
				string(0) ""
				["castrationTechnique"]=>
				string(0) ""
				["castrationDate"]=>
				string(0) ""
				["dehorningTechnique"]=>
				string(0) ""
				["dehorningDate"]=>
				string(0) ""
				["spayingTechnique"]=>
				string(0) ""
				["spayingDate"]=>
				string(0) ""
				["bodyTemperature"]=>
				string(0) ""
				["notes"]=>
				string(0) ""
				}
			}
		$request[2] is the object array for additionalHealth (the four treatments and four vaccines), each
			value for the keys hold an array of IDs for animals which will have either a treatment or vaccine
			  [2]=>
				array(8) {
					[0]=>
					array(1) {
					[0]=>
					int(878)
					}
					[1]=>
					array(0) {
					}
					[2]=>
					array(0) {
					}
					[3]=>
					array(0) {
					}
					[4]=>
					array(0) {
					}
					[5]=>
					array(0) {
					}
					[6]=>
					array(0) {
					}
					[7]=>
					array(0) {
					}
				}
		$request[3] is an object array with the selected treatment/vaccination details:
			  [3]=>
				array(1) {
					[0]=>
					object(stdClass)#3 (5) {
					["value"]=>
					string(11) "Treatment 1"
					["isChecked"]=>
					bool(true)
					["type"]=>
					string(3) "all"
					["text"]=>
					string(54) "Bones/Muscles - Injury- leg; Actogain - As an epidural"
					["index"]=>
					int(0)
					}
				}
		$request[4] is the actual treatment/vaccination data
		  [4]=>
			array(1) {
				[0]=>
				object(stdClass)#4 (11) {
				["date"]=>
				string(10) "2023-06-22"
				["volume"]=>
				string(0) ""
				["units"]=>
				string(0) ""
				["product"]=>
				string(8) "Actogain"
				["productOther"]=>
				string(0) ""
				["route"]=>
				string(1) "4"
				["reason"]=>
				string(1) "3"
				["system"]=>
				string(1) "2"
				["condition"]=>
				string(1) "8"
				["weight"]=>
				string(0) ""
				["notes"]=>
				string(0) ""
				}
			}
		*/

		// integer: 5 for `custom` (custom processing form)
		$actionTypeID = intval($mysqli->real_escape_string($request[0]->actionTypeID));
		// e.g., string: `custom`
		$action = $mysqli->real_escape_string($request[0]->action);
		// e.g., string: `animals` or `groups`
		$itemType = $mysqli->real_escape_string($request[0]->itemType);
		// e.g., string: `single` or `multiple`
		$selectedAmount = $mysqli->real_escape_string($request[0]->selectedAmount);
		$IDs = $request[0]->customIDs;
		// e.g., string: `ext-kn1`
		$userID = $mysqli->real_escape_string($request[0]->userID);
		// e.g., boolean: true
		$loggedIn = $request[0]->loggedIn;

		if ($userID && $loggedIn) {
			// $request[1] = formDataList, list of form dictionaries
			if ($request[1]) {
				// Initialize array to hold form data arrays (array of arrays)
				$allFormData = array();

				// Iterate through each of the passed form dictionaries from the list
				foreach ($request[1] as $r) {
					// Initialize array to hold current form data
					$formData = array();

					// Iterate through the current form data as key/value pairs
					foreach ($r as $key => $value) {
						/*
                        * The "ID" key was added to the form data to keep track of item's input data
                            when the user would navigate between them using the added "Tap to Select" button.
                            That way, if the user came across a selected item again, the data would load.
                        */
						if ($key == 'id') {
							continue;
						} elseif (empty($value) || $value === 'null') {
							/*
                            * Check to see if value is empty or string "null"
                            * If so, append null to array
                            */
							$formData[] = null;
						} else {
							/*
                            * Otherwise, append the value after escaping string
                            * Escape special characters in a string for use in an SQL statement,
                                taking into account the current charset of the connection
                            */
							$formData[] = $mysqli->real_escape_string($value);
						}
					}

					// Push the current data array into the list of all data
					$allFormData[] = $formData;
				}
			} else {
				echo json_encode('No form data!');
				exit(1);
			}

			/*
            * If there is data (extracted from passed form data list of form dictionaries)
                and the length of items from the list of form dictionaries equals to length of the list of arrays
            */
			if ($allFormData && count($request[1]) == count($allFormData)) {
				$actionID = getCustomActionID($mysqli, $actionTypeID, $action, $itemType, $selectedAmount);

				if ($actionID) {
					// $key is index of current animal in list of IDs
					// $id will be ID of current item (i.e., animal)
					foreach ($IDs as $key => $id) {
						if ($action === 'custom') {
							// Insert a record into the event_custom table
							$customEventID = customEvent($mysqli, $id, $actionTypeID, $actionID, $itemType, $userID, $allFormData[$key]);
							// Take the returned ID and create an object associating the custom event ID with the animal ID
							$customObject = [
								'ID' => $id,
								'customEventID' => $customEventID
							];
							// Append the current custom object to the list of all custom objects to be used later to reference records
							$allCustomObjects[] = $customObject;
						} else {
							echo json_encode('No Custom Processing Form Action!');
							exit(1);
						}
					}
				} else {
					echo json_encode('No Custom Processing Form Action ID!');
					exit(1);
				}
			} else {
				echo json_encode('No Custom Processing Form data!');
				exit(1);
			}

			/*
			* $request[2] = additionalOptions, list of lists containing IDs for selected items which have an additional option (treatment or vaccination applied)
			* $request[3] = filteredAdditional (options), help identify which option corresponds to which set of IDs
			* $request[4] = actionsData, this corresponds to a list of objects representing
			* selected Treatments or Vaccinations applied to selected items
			* Check to see that $request[2], $request[3], and $request[4] all have data before proceeding
			*/
			if (!empty($request[2]) && !empty($request[3]) && !empty($request[4])) {
				/* Iterate through the additional options IDs list of lists, which holds the
				* item IDs for which to apply a Treatment or Vaccination
				* $request[2] contains 8 object arrays with IDs for animals which receive the Treatment or Vaccination
				*/
				foreach ($request[2] as $key => $value) {
					$currentAnimalIDs = $value;

					// If there is a current list of animal IDs and a defined item type ("animals" or "groups"), then proceed
					if ($currentAnimalIDs && $itemType) {
						// If the current filtered additional option value contains "Treatment", then process the animals as a Health Event
						if (isset($request[3][$key]) && strpos($request[3][$key]->value, 'Treatment') !== false && !empty($request[4][$key])) {
							// Define an array to hold the object values for the current Health data object
							$currentHealthData = [];

							/*
							* Iterate through the current Health data object and push the values into an array
							* This array will be used to process the IDs associated with
							* the Health data (Treatment) at the specific index
							*/
							foreach ($request[4][$key] as $k => $v) {
								/*
								* The `volume` input field was made optional since defining an additional option
								* within the custom form applies to all selected items
								* So, if the user selected a volume amount, then that would
								* apply to all items, which is not the case
								* Set an empty `volume` value to NULL
								*/
								if ($v === '') {
									$v = null;
								}

								$currentHealthData[] = $v;
							}

							// Set appropriate variables related to action
							$actionTypeID = 2; // 2 = health (event)

							if ($itemType === 'animals') {
								$actionID = 9; // 9 = health (treatment) for single animal
							} else if ($itemType === 'groups') {
								$actionID = 11; // 11 = health (treatment) for single group
							}

							// Iterate through the IDs and process Health (Treatment) events
							foreach ($currentAnimalIDs as $id) {
								$healthRecordID = treatmentHealthEvent($mysqli, $id, $actionTypeID, $actionID, $itemType, $userID, $currentHealthData);

								// Find the customEventID associated with the current ID
								$customEventID = null;
								foreach ($allCustomObjects as $customObject) {
									if ($customObject['ID'] == $id) {
										$customEventID = $customObject['customEventID'];
										break;
									}
								}

								// Check if the customEventID was found
								if ($customEventID !== null) {
									// Insert a new record into event_health_mapping
									insertHealthMappingRecord($mysqli, $healthRecordID, $customEventID);
								}
							}
						} elseif (isset($request[3][$key]) && strpos($request[3][$key]->value, 'Vaccination') !== false && !empty($request[4][$key])) {
							/* Else, if the current filtered additional option value contains "Vaccination",
							* then process the animals as a Vaccination Event
							*/

							// Define an array to hold the object values for the current Vaccination data object
							$currentVaccinationData = [];

							/*
							* Iterate through the current Vaccination data object and push the values into an array
							* This array will be used to process the IDs associated with the Vaccination data at the specific index
							*/
							foreach ($request[4][$key] as $k => $v) {
								/*
								* The `volume` input field was made optional since defining an additional
								* option within the custom form applies to all selected items
								* So, if the user selected a volume amount, then that would apply to
								* all items, which is not the case
								* Set an empty `volume` value to NULL
								* Additionally, the `serial number` and `expiration date` fields were
								* originally optional, convert their empty values to NULL
								*/
								if ($v === '') {
									$v = null;
								}

								$currentVaccinationData[] = $v;
							}

							// Set appropriate variables related to the current action
							$actionTypeID = 3; // 3 = vaccination (event)

							if ($itemType === 'animals') {
								$actionID = 13; // 13 = vaccination for single animal
							} else if ($itemType === 'groups') {
								$actionID = 15; // 15 = vaccination for single group
							}

							// Iterate through the IDs and process Vaccination events
							foreach ($currentAnimalIDs as $id) {
								$vaccinationRecordID = vaccinationEvent($mysqli, $id, $actionTypeID, $actionID, $itemType, $userID, $currentVaccinationData);

								// Find the customEventID associated with the current ID
								$customEventID = null;
								foreach ($allCustomObjects as $customObject) {
									if ($customObject['ID'] == $id) {
										$customEventID = $customObject['customEventID'];
										break;
									}
								}

								// Check if the customEventID was found
								if ($customEventID !== null) {
									// Insert a new record into event_health_mapping
									insertVaccinationMappingRecord($mysqli, $vaccinationRecordID, $customEventID);
								}
							}
						}
					}
				}
			}
		} else {
			echo json_encode('Not logged in!');
			$mysqli->close();
			exit(1);
		}

		echo json_encode('success');
		$mysqli->close();
	} else {
		echo json_encode('No data!');
		exit(1);
	}
} else {
	echo json_encode('Invalid Custom Processing Form Event!');
	exit(1);
}
