<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require('accessControl.php');

/*
BY:				krn
FILENAME:		actionsCustomProcessingFormEvent.php
DESCRIPTION:	This PHP script manages custom processing form events by defining arrays for custom, health,
				and vaccination objects. It includes functions for updating custom events, checking IDs,
				creating vaccination and treatment health events, creating custom events, and retrieving the
				custom action ID. Upon receiving a POST request with specific data, the script
				validates user authentication and processes the custom form data. It inserts custom events
				into the database and handles additional options like treatments and vaccinations.
				The script updates the associated animal IDs in custom event records, and returns a
				JSON response indicating success or failure before closing the database connection.
*/

// define local time zone and format
date_default_timezone_set('America/Chicago');

/**
 * Helper function to send and log error messages
 * 
 * @param string $message The error message to send and log
 *
 * @return void
 */
function handleError(string $message): void
{
	// Send error to the client
	echo json_encode($message);

	// Log error on the server
	error_log($message);

	// Exit the script
	exit(1);
}


/**
 * Find the custom event ID associated with an animal.
 *
 * @param int   $animalID          ID of the animal.
 * @param array $allCustomObjects  Array of custom event objects.
 *
 * @return int                    The custom event ID if found.
 * @throws Exception              If the custom event ID is not found.
 */
function findCustomEventIDForAnimal(int $animalID, array $allCustomObjects): int
{
	foreach ($allCustomObjects as $customObject) {
		if ($customObject['ID'] === $animalID) {
			return $customObject['customEventID'];
		}
	}

	// If no matching custom event ID found, throw an error and exit
	throw new Exception("findCustomEventIDForAnimal Custom event ID not found for animal ID: $animalID");
}


/**
 * Inserts a mapping record for health events.
 *
 * @param mysqli $mysqli    The MySQLi object representing the database connection.
 * @param int    $healthID  ID of the health event.
 * @param int    $eventID   ID of the associated event.
 *
 * @return void
 */
function insertHealthMappingRecord(mysqli $mysqli, int $healthID, int $eventID): void
{
	try {
		// Prepare the INSERT query for the event_health_mapping table
		$query = $mysqli->prepare('INSERT INTO `event_health_mapping` (`health_id`, `event_id`, `event_datetime`) VALUES (?, ?, ?)');

		// Get the current date and time in 'Y-m-d H:i:s' format
		$eventDateTime = date('Y-m-d H:i:s');

		// Check if the query was prepared successfully
		if ($query) {
			// Bind the parameters to the prepared statement
			$query->bind_param('iis', $healthID, $eventID, $eventDateTime);

			// Execute the prepared statement
			$query->execute();

			// Close the statement
			$query->close();
		} else {
			// Output the error as a JSON string and exit
			handleError("insertHealthMappingRecord Error: " . $mysqli->error);
		}
	} catch (Exception $e) {
		// Handle exceptions here, e.g., log the error or terminate gracefully
		handleError("insertHealthMappingRecord Error: " . $e->getMessage());
	}
}


/**
 * Inserts a mapping record for vaccination events.
 *
 * @param mysqli $mysqli        The MySQLi object representing the database connection.
 * @param int    $vaccinationID ID of the vaccination event.
 * @param int    $eventID       ID of the associated event.
 *
 * @return void
 */
function insertVaccinationMappingRecord(mysqli $mysqli, int $vaccinationID, int $eventID): void
{
	try {
		// Prepare the INSERT query for the event_vaccination_mapping table
		$query = $mysqli->prepare('INSERT INTO `event_vaccination_mapping` (`vaccination_id`, `event_id`, `event_datetime`) VALUES (?, ?, ?)');

		// Get the current date and time in 'Y-m-d H:i:s' format
		$eventDateTime = date('Y-m-d H:i:s');

		// Check if the query was prepared successfully
		if ($query) {
			// Bind the parameters to the prepared statement
			$query->bind_param('iis', $vaccinationID, $eventID, $eventDateTime);

			// Execute the prepared statement
			$query->execute();

			// Close the statement
			$query->close();
		} else {
			// Output the error as a JSON string and exit
			handleError("insertVaccinationMappingRecord Error: " . $mysqli->error);
		}
	} catch (Exception $e) {
		// Handle exceptions here, e.g., log the error or terminate gracefully
		handleError("insertVaccinationMappingRecord Error: " . $e->getMessage());
	}
}


/**
 * Creates a vaccination event record.
 *
 * @param mysqli $mysqli         	The MySQLi object representing the database connection.
 * @param int    $id             	ID of the item (animal or group).
 * @param int    $actionTypeID   	Action type ID.
 * @param int    $actionID       	Action ID.
 * @param string $userID         	User ID.
 * @param array  $vaccinationData 	Array containing vaccination event data.
 *
 * @return int                   	Returns the last inserted ID from the `event_vaccination` table.
 */
function vaccinationEvent(mysqli $mysqli, int $id, int $actionTypeID, int $actionID, string $userID, array $vaccinationData): int
{

	try {
		// Prepare the INSERT query for the event_vaccination table
		$query = $mysqli->prepare(
			'INSERT INTO `event_vaccination`
            (`action_type_id`, `action_id`, `animal_id`, `date`, `volume`, `units_id`, `vaccine`, `vaccine_other`,
            `roa_id`, `serial_number`, `expiration_date`, `user_id`, `event_datetime`)
            VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?)'
		);

		// Get the current date and time in 'Y-m-d H:i:s' format
		$eventDateTime = date('Y-m-d H:i:s');

		// Since expiration date is a date type, we need to set NULL value if empty string ""
		if (empty($vaccinationData['expirationDate'])) {
			$vaccinationData['expirationDate'] = NULL;
		}

		if (
			$query &&
			$query->bind_param(
				'iiisdississss',
				$actionTypeID,    						// action type ID
				$actionID,        						// action ID
				$id,              						// ID of item (animal or group)
				$vaccinationData['date'], 				// date
				$vaccinationData['volume'], 			// volume (optional)
				$vaccinationData['units'], 				// units
				$vaccinationData['vaccine'], 			// vaccine
				$vaccinationData['vaccineOther'], 		// vaccine other (if the user selects "Other" as vaccine and inputs another vaccine)
				$vaccinationData['route'], 				// route of administration
				$vaccinationData['serialNumber'], 		// serial number (optional)
				$vaccinationData['expirationDate'], 	// expiration date (optional)
				$userID,          						// user ID
				$eventDateTime    						// database record date
			) &&
			$query->execute()
		) {
			$query->close();
			return $mysqli->insert_id;
		} else {
			handleError("vaccinationEvent Error: " . $mysqli->error);
		}
	} catch (Exception $e) {
		// Handle exceptions here, e.g., log the error or terminate gracefully
		handleError("vaccinationEvent Error: " . $e->getMessage());
	}
}


/**
 * Creates a health event record for treatment.
 *
 * @param mysqli $mysqli         The MySQLi object representing the database connection.
 * @param int    $id             ID of the item (animal or group).
 * @param int    $actionTypeID   Action type ID.
 * @param int    $actionID       Action ID.
 * @param string $userID         User ID.
 * @param array  $healthData     Array containing health event data.
 *
 * @return int                   Returns the last inserted ID from the `event_health` table.
 */
function treatmentHealthEvent(mysqli $mysqli, int $id, int $actionTypeID, int $actionID, string $userID, array $healthData): int
{

	try {
		// Prepare the INSERT query for the event_health table
		$query = $mysqli->prepare(
			'INSERT INTO `event_health`
            (`action_type_id`, `action_id`, `animal_id`, `date`, `volume`, `units_id`, `product`, `product_other`,
            `roa_id`, `reason`, `problem_bs`, `problem_bsc`, `weight`, `notes`, `user_id`, `event_datetime`)
            VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)'
		);

		// Get the current date and time in 'Y-m-d H:i:s' format
		$eventDateTime = date('Y-m-d H:i:s');

		if (
			$query &&
			$query->bind_param(
				'iissiissiiiiisss',
				$actionTypeID,    				// action type ID
				$actionID,        				// action ID
				$id,              				// ID of item (animal or group)
				$healthData['date'],     		// date
				$healthData['volume'],   		// volume (optional)
				$healthData['units'],    		// units
				$healthData['product'],  		// product
				$healthData['productOther'], 	// product other (if the user selects "Other" as product and inputs another product)
				$healthData['route'],    		// route of administration
				$healthData['reason'],   		// reason
				$healthData['system'],   		// problem body system
				$healthData['condition'],		// problem body system condition				
				$healthData['weight'],			// weight (optional)
				$healthData['notes'],			// notes (optional)
				$userID,          				// user ID
				$eventDateTime    				// database record date
			) &&
			$query->execute()
		) {
			$query->close();
			return $mysqli->insert_id;
		} else {
			handleError("treatmentHealthEvent Error: " . $mysqli->error);
		}
	} catch (Exception $e) {
		// Handle exceptions here, e.g., log the error or terminate gracefully
		handleError("treatmentHealthEvent Error: " . $e->getMessage());
	}
}


/**
 * Process animals with custom events.
 *
 * @param mysqli $mysqli          		The MySQLi object representing the database connection.
 * @param array  $animals         		Array of animals to be processed.
 * @param array  $allCustomObjects 		Array of custom objects for animals.
 * @param array  $additionalOptions 	Array of additional options.
 * @param int    $actionTypeID    		Action type ID.
 * @param int    $actionID        		Action ID.
 * @param string $userID          		User ID.
 */
function processAnimalsWithCustomEvents(mysqli $mysqli, array $animals, array $allCustomObjects, array $additionalOptions, int $actionTypeID, int $actionID, string $userID): void
{
	try {
		foreach ($animals as $animal) {

			$id = $animal->id;
			$customEventID = findCustomEventIDForAnimal($id, $allCustomObjects);

			// Handle Health/Treatment Events based on selectedTreatments
			if (!empty($animal->selectedTreatments)) {
				foreach ($animal->selectedTreatments as $index) {
					$healthData = $additionalOptions[$index]->data;
					$healthEventID = treatmentHealthEvent($mysqli, $id, $actionTypeID, $actionID, $userID, (array) $healthData);
					insertHealthMappingRecord($mysqli, $healthEventID, $customEventID);
				}
			}

			// Handle Vaccination Events based on selectedVaccinations
			if (!empty($animal->selectedVaccinations)) {
				foreach ($animal->selectedVaccinations as $index) {
					$vaccinationData = $additionalOptions[$index]->data;
					$vaccinationEventID = vaccinationEvent($mysqli, $id, $actionTypeID, $actionID, $userID, (array) $vaccinationData);
					insertVaccinationMappingRecord($mysqli, $vaccinationEventID, $customEventID);
				}
			}
		}
	} catch (Exception $e) {
		// Handle exceptions here, e.g., log the error or terminate gracefully
		handleError("processAnimalsWithCustomEvents Error: " . $e->getMessage());
	}
}


/**
 * Inserts a custom event record into the `event_custom` table.
 *
 * @param mysqli $mysqli         The MySQLi object representing the database connection.
 * @param int    $id             ID of the item (animal or group).
 * @param int    $actionTypeID   Action type ID.
 * @param int    $actionID       Action ID.
 * @param string $userID         User ID.
 * @param array  $customData     Array containing all custom event data.
 *
 * @return int                   Returns the last inserted ID from the `event_custom` table.
 */
function customEvent(mysqli $mysqli, int $id, int $actionTypeID, int $actionID, string $userID, array $customData): int
{
	try {
		$insertCustomEventQuery = $mysqli->prepare(
			'INSERT INTO `event_custom` 
            (`action_type_id`, `action_id`, `animal_id`, `weight`, `weight_id`, 
            `hip_height`, `bcs`, `repro_tract_score`, `chute_docility_score`, `foot_angle_score`, 
            `claw_set_score`, `teat_score`, `udder_score`, `bse_motility`, `bse_morphology`, 
            `bse_acceptable`, `pregnancy_status`, `gestation_length`, `gl_units`, `castration_tech`, 
            `castration_date`, `dehorning_tech`, `dehorning_date`, `spaying_tech`, `spaying_date`, 
            `body_temperature`, `notes`, `user_id`, `event_datetime`)
            VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)'
		);

		$eventDateTime = date('Y-m-d H:i:s');

		if (
			$insertCustomEventQuery &&
			$insertCustomEventQuery->bind_param(
				'iiididiiiiiiiiiisssisisisssss',
				$actionTypeID,      // action type ID
				$actionID,          // action ID
				$id,                // ID of item (animal or group)
				$customData[0],     // weight
				$customData[1],     // weight id
				$customData[2],     // hip height
				$customData[3],     // body condition score (bcs)
				$customData[4],     // reproductive tract score (cows only)
				$customData[5],     // chute docility score
				$customData[6],     // foot angle score
				$customData[7],     // claw set score
				$customData[8],     // teat score (cows only)
				$customData[9],    // udder score (cows only)
				$customData[10],    // bse motility (breeding soundness exam) (bulls only)
				$customData[11],    // bse morphology (breeding soundness exam) (bulls only)
				$customData[12],    // bse acceptable (breeding soundness exam) (bulls only)
				$customData[13],    // pregnancy status (cows only)
				$customData[14],    // pregnancy gestation length (cows only)
				$customData[15],    // pregnancy gestation length units (cows only)
				$customData[16],    // castration technique (bulls only)
				$customData[17],    // castration date (bulls only)
				$customData[18],    // dehorning technique (all)
				$customData[19],    // dehorning date (all)
				$customData[20],    // spaying technique (cows only)
				$customData[21],    // spaying date (cows only)
				$customData[22],    // body temperature
				$customData[23],    // notes
				$userID,            // user ID
				$eventDateTime      // database record date
			) &&
			$insertCustomEventQuery->execute()
		) {
			$insertCustomEventQuery->close();
			return $mysqli->insert_id;
		} else {
			handleError("customEvent Error: " . $mysqli->error);
		}
	} catch (Exception $e) {
		// Handle exceptions here, e.g., log the error or return a default value
		handleError("customEvent Error: " . $e->getMessage());
	}
}


/**
 * Process custom events for animals.
 *
 * @param mysqli $mysqli          The MySQLi object representing the database connection.
 * @param array  $allFormData     All form data.
 * @param int    $actionTypeID    Action type ID.
 * @param string $action          Action string.
 * @param string $itemType        Item type ('animal' or 'group').
 * @param string $selectedAmount  Selected amount ('single' or 'multiple').
 * @param string $userID          User ID.
 * @param array  $animals         Array of animals to be processed.
 *
 * @return array                  Returns an array with actionID and all custom objects.
 */
function processCustomEventsForAnimals(mysqli $mysqli, array $allFormData, int $actionTypeID, string $action, string $itemType, string $selectedAmount, string $userID, array $animals): array
{
	$allCustomObjects = [];
	$actionID = null;

	try {
		// Attempt to get the custom action ID
		$actionID = getCustomActionID($mysqli, $actionTypeID, $action, $itemType, $selectedAmount);

		// If $actionID is not NULL, proceed to process the animals
		if ($actionID !== null) {
			foreach ($animals as $key => $animal) {
				$id = $animal->id;
				if ($action === 'custom') {
					$customEventID = customEvent($mysqli, $id, $actionTypeID, $actionID, $userID, $allFormData[$key]);

					$customObject = [
						'ID' => $id,
						'customEventID' => $customEventID
					];
					$allCustomObjects[] = $customObject;
				} else {
					handleError('No Custom Processing Form Action!');
				}
			}
		} else {
			// Handle the situation where $actionID is NULL
			handleError("processCustomEventsForAnimals NULL Action ID");
		}
	} catch (Exception $e) {
		handleError("processCustomEventsForAnimals Error: " . $e->getMessage());
	}

	return [
		'actionID' => $actionID,
		'allCustomObjects' => $allCustomObjects
	];
}



/**
 * Get the custom action ID from the database.
 *
 * @param mysqli $mysqli         The MySQLi object representing the database connection.
 * @param int    $actionTypeID   Action type ID.
 * @param string $action         Action string.
 * @param string $itemType       Item type ('animal' or 'group').
 * @param string $selectedAmount Selected amount ('single' or 'multiple').
 *
 * @return int                   Returns the action ID from the database.
 * @throws Exception            Throws an exception if no matching record is found.
 */
function getCustomActionID($mysqli, int $actionTypeID, string $action, string $itemType, string $selectedAmount)
{
	try {
		$selectActionIDQuery = $mysqli->prepare(
			'SELECT `id`
            FROM `actions`
            WHERE `action_type_id` = ?
            AND `action` = ?
            AND `item` = ?
            AND `amount` = ?'
		);

		$actionID = 0; // Initialize $actionID

		if (
			!$selectActionIDQuery ||
			!$selectActionIDQuery->bind_param('isss', $actionTypeID, $action, $itemType, $selectedAmount) ||
			!$selectActionIDQuery->execute()
		) {
			// Handle database query error
			throw new Exception("getCustomActionID Database query error: " . $mysqli->error);
		}

		$selectActionIDQuery->store_result();
		$selectActionIDQuery->bind_result($actionID);

		if (!$selectActionIDQuery->fetch()) {
			// No matching record found, throw an exception
			throw new Exception("getCustomActionID No matching record found for actionTypeID $actionTypeID and action $action");
		}

		$selectActionIDQuery->close();

		return $actionID;
	} catch (Exception $e) {
		// Handle exceptions here, e.g., log the error or return a default value
		handleError("getCustomActionID Error: " . $e->getMessage());
	}
}


/**
 * Extract and process form data.
 *
 * @param array  $formDataList List of form data dictionaries.
 * @param mysqli $mysqli      The MySQLi object representing the database connection.
 *
 * @return array              Returns an array of processed form data arrays.
 */
function extractFormData(array $formDataList, mysqli $mysqli): array
{
	try {
		if (!$formDataList) {
			handleError('No form data!');
		}

		// Holds processed form data arrays.
		$allFormData = [];

		// Process each form dictionary in the list.
		foreach ($formDataList as $form) {
			$formData = []; // Stores processed data for the current form.

			// Process each key/value pair in the form.
			foreach ($form as $key => $value) {
				// Skip the 'id' key as it's used for internal tracking and not for database insertion.
				if ($key === 'id') {
					continue;
				}

				if (empty($value) || $value === 'null') {
					// If value is empty or string "null", append null to array.
					$formData[] = null;
				} else {
					// Otherwise, append cleaned value
					$trimmedValue = trim($value);
					$escapedValue = $mysqli->real_escape_string($trimmedValue);
					$formData[] = $escapedValue;
				}
			}

			// Append the processed data for the current form to the master list.
			$allFormData[] = $formData;
		}

		return $allFormData;
	} catch (Exception $e) {
		// Handle exceptions here, e.g., log the error or return a default value
		handleError("extractFormData Error: " . $e->getMessage());
	}
}


/**
 * Process the custom form data from a POST request.
 */
function processCustomFormRequest(): void
{
	try {
		// Check if the request method is POST.
		if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
			handleError('Invalid Custom Processing Form Event!');
		}

		$data = file_get_contents('php://input');
		$request = json_decode($data);

		if (!isset($request[0], $request[1])) {
			handleError('No data!');
		}

		require('connection.php');

		/*
		$request[0] is an object of passed data relevant to the action performed
		object(stdClass)#1 (6) {
			["actionTypeID"]=> int(4)
			["action"]=> string(6) "custom"
			["itemType"]=> string(7) "animals"
			["selectedAmount"]=> string(8) "multiple"
			["userID"]=> string(8) "ext-ct38"
			["loggedIn"]=> bool(true)
		}

		$request[1] is an array of objects representing form data relevant to the action performed
		For example, if the user has selected 3 animals for the custom form, then $request[1]
		should be an array of 3 elements, objects with (26 keys)
		array(3) {
			[0]=>
			object(stdClass)#2 (26) {
				["id"]=> int(878)
				["weaningWeight"]=> int(12)
				["yearlingWeight"]=> int(34)
				["matureWeight"]=> string(0) "
				["hipHeight"]=> string(0) "
				["bodyConditionScore"]=> string(0) "
				["reproductiveTractScore"]=> string(0) "
				["chuteDocilityScore"]=> string(0) "
				["footAngleScore"]=> string(0) "
				["clawSetScore"]=> string(0) "
				["teatScore"]=> string(0) "
				["udderScore"]=> string(0) "
				["bseMotility"]=> string(0) "
				["bseMorphology"]=> string(0) "
				["bseAcceptableBreeder"]=> string(0) "
				["pregnancyStatus"]=> string(0) "
				["gestationLength"]=> string(0) "
				["gestationLengthUnits"]=> string(0) "
				["castrationTechnique"]=> string(0) "
				["castrationDate"]=> string(0) "
				["dehorningTechnique"]=> string(0) "
				["dehorningDate"]=> string(0) "
				["spayingTechnique"]=> string(0) "
				["spayingDate"]=> string(0) "
				["bodyTemperature"]=> string(0) "
				["notes"]=> string(0) "
			}
			...
		}

		$request[2] is an array of objects with data corresponding to selected animals
		In this data we are only concerned with the "id", "selectedTreatments" and "selectedVaccinations" fields
		The latter contain lists which correspond to the indices of the passed additionalOptions data, present in $request[3]
		However, we are passing the whole animal object. This may be condensed at a later time as needed.
		Again, if the user has selected 3 animals for the custom form, then we will have an array of 3 objects (40 keys)
		array(3) {
			[0]=>
			object(stdClass)#5 (40) {
				["id"]=> int(878)
				["primary_id"]=> string(4) "1609"
				["sex_id"]=> int(1)
				["addition_type"]=> int(2)
				["date_birth"]=> string(10) "2011-10-15"
				["date_purchase"]=> NULL
				["breed_id"]=> int(15)
				["color_id"]=> int(1)
				["weight"]=> NULL
				["tag_color"]=> NULL
				["official_nues"]=> NULL
				["official_ain"]=> NULL
				["twin_status_id"]=> NULL
				["calving_vigor_score_id"]=> NULL
				["dam"]=> NULL
				["teat_score_id"]=> NULL
				["udder_score_id"]=> NULL
				["calving_ease_id"]=> NULL
				["dam_bcs_id"]=> NULL
				["sire"]=> NULL
				["status_id"]=> int(1)
				["reason_id"]=> int(1)
				["cause_id"]=> int(1)
				["notes"]=> string(0) "
				["user_id"]=> string(8) "ext-ct38"
				["record_type_id"]=> int(1)
				["date_created"]=> string(19) "2022-12-07 15:22:31"
				["date_updated"]=> string(19) "2022-12-07 15:22:31"
				["sex_name"]=> string(4) "Bull"
				["inactive_reason"]=> string(32) "Animal is active (default value)"
				["inactive_cause"]=> string(32) "Animal is active (default value)"
				["group_names"]=> string(10) "Herd bulls"
				["group_ids"]=> string(2) "56"
				["category_names"]=> string(19) "Spring calving herd"
				["category_ids"]=> string(2) "31"
				["dam_name"]=> NULL
				["sire_name"]=> NULL
				["isChecked"]=> bool(true)
				["selectedTreatments"]=> array(1) { [0]=> int(4) }
				["selectedVaccinations"]=>array(0) {}
			}
		...
		}

		$request[3] correspond to the additionalOptions set within the app on the custom form
		It is a pre-defined array containing 8 objects; 4 treatments and 4 vaccines
		The values within the prior $request[2] (selectedAnimals) keys of "selectedTreatments" and "selectedVaccinations" correspond
		to the indices of the array object. That is, if animal A has selectedTreatments=[4], then 4 means index 4 of additionalOptions which is "Vaccination 1"
		We are primarily concerned with the data associated with these objects containing health data such as date, product, route, etc.
		array(8) {
			[0]=>
			object(stdClass)#8 (7) {
				["value"]=> string(11) "Treatment 1"
				["initialChecked"]=> bool(false)
				["isChecked"]=> bool(false)
				["type"]=> string(3) "all"
				["text"]=> string(0) "
				["index"]=> int(-1)
				["data"]=> object(stdClass)#9 (11) {
				["date"]=> string(0) "
				["volume"]=> string(0) "
				["units"]=> string(0) "
				["product"]=> string(0) "
				["productOther"]=> string(0) "
				["route"]=> string(0) "
				["reason"]=> string(0) "
				["system"]=> string(0) "
				["condition"]=> string(0) "
				["weight"]=> string(0) "
				["notes"]=> string(0) "
				}
			}
			...
			[4]=>
			object(stdClass)#16 (7) {
				["value"]=> string(13) "Vaccination 1"
				["initialChecked"]=> bool(true)
				["isChecked"]=> bool(true)
				["type"]=> string(3) "all"
				["text"]=> string(45) "Bovilis® Vista® BVD CFP; Intravenously (IV)"
				["index"]=> int(-1)
				["data"]=> object(stdClass)#17 (8) {
				["date"]=> string(10) "2023-01-01"
				["volume"]=> string(0) "
				["units"]=> string(1) "5"
				["vaccine"]=> string(25) "Bovilis® Vista® BVD CFP"
				["vaccineOther"]=> string(0) "
				["route"]=> string(1) "3"
				["serialNumber"]=> string(0) "
				["expirationDate"]=> string(0) "
				}
			}
		}
		*/

		$actionTypeID = intval($mysqli->real_escape_string($request[0]->actionTypeID)); // integer, 4
		$action = $mysqli->real_escape_string($request[0]->action); // string "custom"
		$itemType = "animals";
		$selectedAmount = $mysqli->real_escape_string($request[0]->selectedAmount); // string, "single" or "multiple"
		$userID = $mysqli->real_escape_string($request[0]->userID); // string "ext-" prefix
		$loggedIn = $request[0]->loggedIn; // boolean; true / false

		$formDataList = $request[1];
		$animals = $request[2];
		$additionalOptions = $request[3];

		if (!$userID || !$loggedIn) {
			handleError('Not logged in!');
			$mysqli->close();
		}

		$allFormData = extractFormData($formDataList, $mysqli);

		if (!$allFormData || count($formDataList) !== count($allFormData)) {
			handleError('No Custom Processing Form data!');
		}

		// Start a transaction
		$mysqli->begin_transaction();

		try {
			$result = processCustomEventsForAnimals($mysqli, $allFormData, $actionTypeID, $action, $itemType, $selectedAmount, $userID, $animals);
			$actionID = $result['actionID'];
			$allCustomObjects = $result['allCustomObjects'];

			processAnimalsWithCustomEvents($mysqli, $animals, $allCustomObjects, $additionalOptions, $actionTypeID, $actionID, $userID);

			// If everything is successful, commit the transaction
			$mysqli->commit();

			handleError('success');
		} catch (Exception $e) {
			// If an error occurs, roll back the transaction
			$mysqli->rollback();
			handleError('Rollback From Error:', $e->getMessage());
		} finally {
			$mysqli->close();
		}
	} catch (Exception $e) {
		// Handle exceptions here, e.g., log the error or terminate gracefully
		handleError("processCustomFormRequest Error: " . $e->getMessage());
	}
}

// Execute the function to process the form request.
processCustomFormRequest();
