<?php
require('accessControl.php');

/*
BY:             krn
FILENAME:       actionsHealthEvent.php
DESCRIPTION:    When the user is recording data for active animals, they have the option to add a health event.
                This health event can correspond to an observation (no treatment) or a treatment.
                Add a new health event record in the database depending on the user's selection.
                Both types of health events have required and optional data.
*/

// define local time zone and format
date_default_timezone_set('America/Chicago');

function treatmentHealthEvent($mysqli, $id, $actionTypeID, $actionID, $itemType, $userID, $healthData)
{
    if ($itemType == 'animals') {
        $insertHealthEventQuery = $mysqli->prepare('INSERT INTO `event_health`
        (`action_type_id`, `action_id`, `animal_id`, `date`, `volume`, `units_id`,
        `product`, `product_other`, `roa_id`, `reason`, `problem_bs`,
        `problem_bsc`, `weight`, `notes`, `user_id`, `event_datetime`)
        VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)');
    } elseif ($itemType === 'groups') {
        $insertHealthEventQuery = $mysqli->prepare('INSERT INTO `event_health`
        (`action_type_id`, `action_id`, `group_id`, `date`, `volume`, `units_id`,
        `product`, `product_other`, `roa_id`, `reason`, `problem_bs`,
        `problem_bsc`, `weight`, `notes`, `user_id`, `event_datetime`)
        VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)');
    } else {
        echo json_encode('Invalid Health (Treatment) Item Type!');
        exit(1);
    }

    // set date created/updated for record insertion
    $eventDateTime = date('Y-m-d H:i:s');

    if (
        $insertHealthEventQuery &&
        $insertHealthEventQuery->bind_param(
            'iiisdissiiiidsss',
            $actionTypeID, // action type ID
            $actionID, // action ID
            $id, // ID of item (animal or group)
            $healthData['date'], // date
            $healthData['volume'], // volume (optional)
            $healthData['units'], // units_id (references units table)
            $healthData['product'], // product
            $healthData['productOther'], // product other (if the user selects "Other" as product and inputs another product)
            $healthData['route'], // roa_id (reference route of administration table)
            $healthData['reason'], // reason (id)
            $healthData['system'], // problem body system (id)
            $healthData['condition'], // problem body system condition (id)
            $healthData['weight'],  // weight (optional)
            $healthData['notes'], // notes (optional)
            $userID, // user ID
            $eventDateTime // database record date
        ) &&
        $insertHealthEventQuery->execute()
    ) {
        $insertHealthEventQuery->close();
        return;
    } else {
        echo json_encode($mysqli->error);
        exit(1);
    }
}

function observationHealthEvent($mysqli, $id, $actionTypeID,  $actionID, $itemType, $userID, $healthData)
{
    if ($itemType == 'animals') {
        $insertHealthEventQuery = $mysqli->prepare('INSERT INTO `event_health`
        (`action_type_id`, `action_id`, `animal_id`, `problem_bs`, `problem_bsc`,
        `date`, `weight`, `notes`, `user_id`, `event_datetime`)
        VALUES (?,?,?,?,?,?,?,?,?,?)');
    } elseif ($itemType === 'groups') {
        $insertHealthEventQuery = $mysqli->prepare('INSERT INTO `event_health`
        (`action_type_id`, `action_id`, `group_id`, `problem_bs`, `problem_bsc`,
        `date`, `weight`, `notes`, `user_id`, `event_datetime`)
        VALUES (?,?,?,?,?,?,?,?,?,?)');
    } else {
        echo json_encode('Invalid Health (Observation) Item Type!');
        exit(1);
    }

    // set date created/updated for record insertion
    $eventDateTime = date('Y-m-d H:i:s');

    if (
        $insertHealthEventQuery &&
        $insertHealthEventQuery->bind_param(
            'iiiiisdsss',
            $actionTypeID, // action type ID
            $actionID, // action ID
            $id, // ID of item (animal or group)
            $healthData['system'], // problem body system (id) [required]
            $healthData['condition'], // problem body system condition (id) [required]
            $healthData['date'], // date of observation (required)
            $healthData['weight'], // weight (optional, can be null)
            $healthData['notes'], // notes (optional, can be null)
            $userID, // user ID
            $eventDateTime // database record date
        ) &&
        $insertHealthEventQuery->execute()
    ) {
        $insertHealthEventQuery->close();
        return;
    } else {
        echo json_encode($mysqli->error);
        exit(1);
    }
}

function getHealthActionID($mysqli, $actionTypeID, $action, $itemType, $selectedAmount)
{
    // Determine the ID associated with the current action based on the action type ID, action, item, and amount
    $selectActionIDQuery = $mysqli->prepare(
        'SELECT `id`
        FROM `actions`
        WHERE `action_type_id` = ?
        AND `action` = ?
        AND `item` = ?
        AND `amount` = ?'
    );

    $actionID = NULL;

    if (
        $selectActionIDQuery &&
        $selectActionIDQuery->bind_param(
            'isss',
            $actionTypeID,
            $action,
            $itemType,
            $selectedAmount
        ) &&
        $selectActionIDQuery->execute()
    ) {
        $selectActionIDQuery->store_result();
        $selectActionIDQuery->bind_result($actionID);
        $selectActionIDQuery->fetch();
        return $actionID;
    } else {
        echo json_encode($mysqli->error);
        exit(1);
    }
}

// make sure we have received a post request from the app
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    $data = file_get_contents('php://input');

    $request = json_decode($data);

    if ($request && $request[0] && $request[1]) {
        require('connection.php');

        // 2 for `health`
        $actionTypeID = intval($mysqli->real_escape_string($request[0]->actionTypeID));
        // e.g., `treatment` or `observation`
        $action = $mysqli->real_escape_string($request[0]->action);
        // `animals` or `groups`
        $itemType = $mysqli->real_escape_string($request[0]->itemType);
        $selectedAmount = $mysqli->real_escape_string($request[0]->selectedAmount);
        $IDs = $request[0]->healthIDs;
        $userID = $mysqli->real_escape_string($request[0]->userID);
        $loggedIn = $request[0]->loggedIn;

        /*
        observation (no treatment) form
        system: $healthData['system']
        condition: $healthData['condition']
        date: $healthData['date']
        weight: $healthData['weight']
        notes: $healthData['notes']
        */

        /*
        observation and treatment form
        date: $healthData['date']
        volume: $healthData['volume']
        unit: $healthData['units']
        product: $healthData['product']
        product (other): $healthData['productOther']
        route: $healthData['route']
        reason: $healthData['reason']
        system: $healthData['system']
        condition: $healthData['condition']
        weight: $healthData['weight']
        notes: $healthData['notes']
        */

        if ($userID && $loggedIn) {
            // $request[1] = formData object
            if ($request[1]) {
                $healthData = array();

                foreach ($request[1] as $key => $value) {
                    // If value is empty or string "null", append null to array.
                    if (empty($value) || $value === 'null') {
                        $healthData[$key] = null;
                    } else {
                        // Otherwise, append cleaned value
                        $trimmedValue = trim($value);
                        $escapedValue = $mysqli->real_escape_string($trimmedValue);
                        $healthData[$key] = $escapedValue;
                    }
                }
            } else {
                echo json_encode('No Health form data!');
                exit(1);
            }

            // If there is health data (extracted from passed form data object)
            if ($healthData) {

                $actionID = getHealthActionID($mysqli, $actionTypeID, $action, $itemType, $selectedAmount);

                if ($actionID) {
                    foreach ($IDs as $id) {
                        if ($action == 'observation') {
                            observationHealthEvent($mysqli, $id, $actionTypeID, $actionID, $itemType, $userID, $healthData);
                        } elseif ($action == 'treatment') {
                            treatmentHealthEvent($mysqli, $id, $actionTypeID, $actionID, $itemType, $userID, $healthData);
                        }
                    }
                } else {
                    echo json_encode('No Health Action ID!');
                    exit(1);
                }
            } else {
                echo json_encode('No Health data!');
                exit(1);
            }
        } else {
            echo json_encode('Not logged in!');
            exit(1);
        }

        echo json_encode('success');
        $mysqli->close();
    } else {
        echo json_encode('No data!');
        exit(1);
    }
} else {
    echo json_encode('Invalid Health Event!');
    exit(1);
}
