<?php
require('accessControl.php');
require('eventLog.php');

/*
BY:             krn
FILENAME:       actionsMoveGroup.php
DESCRIPTION:    
*/

/*
* Check to see if there are multiple records associated an animal to a group
* This happens if the user Moves an animal which is associated with multiple groups into a new group
* We assume that the user wants to move this animal out of all current groups into a new single group
* Delete all excess records associated with the passed group id, animal id and user id.
* (As of September 16, 2020)
*/
function deleteExcess($mysqli, $newGroupID, $currentItemID, $userID)
{
    $selectGroupAnimalsQuery = $mysqli->prepare('SELECT `id` FROM `group_animals` WHERE `group_id` = ? and `animal_id` = ? and `user_id` = ?');

    if (
        $selectGroupAnimalsQuery &&
        $selectGroupAnimalsQuery->bind_param(
            'iis',
            $newGroupID,
            $currentItemID,
            $userID
        ) &&
        $selectGroupAnimalsQuery->execute()
    ) {
        $groupAnimalsResult = $selectGroupAnimalsQuery->get_result();
        $selectGroupAnimalsQuery->close();

        $numberRows = $groupAnimalsResult->num_rows;

        // If there is more than 1 instance (record) of the current animal being associated with the current group for the current user
        // Then delete the excess record
        if ($numberRows > 1) {
            // Set count equal to the number of rows
            $count = $numberRows;

            // Prepare `DELETE` query
            $deleteGroupAnimalQuery = $mysqli->prepare('DELETE FROM `group_animals` WHERE `id` = ?');

            // iterate through the row results and append to results array
            while ($row = $groupAnimalsResult->fetch_assoc()) {
                // If the count is greater than 1, then were are duplicate records
                if ($count > 1) {
                    if (
                        $deleteGroupAnimalQuery &&
                        $deleteGroupAnimalQuery->bind_param(
                            'i',
                            $row['id'],
                        ) &&
                        $deleteGroupAnimalQuery->execute()
                    ) {
                        // Decrement amount of duplicate rows left
                        $count = $count - 1;

                        continue;
                    } else {
                        echo json_encode($mysqli->error);
                        exit(1);
                    }
                } else if ($count == 1) {
                    // If the current count of rows is 1, then all excess records have been deleted
                    // Break out of loop
                    break;
                }
            }

            // Close delete query
            $deleteGroupAnimalQuery->close();
        }
    }
}

/*
* Get all the animals for a old group id
* The animals in this group are being moved to a new group selected by the user
*/
function getOldGroupAnimals($mysqli, $groupID)
{
    $selectGroupAnimalsQuery = $mysqli->prepare('SELECT `animal_id` FROM `group_animals` WHERE `group_id` = ?');

    if (
        $selectGroupAnimalsQuery &&
        $selectGroupAnimalsQuery->bind_param('i', $groupID) &&
        $selectGroupAnimalsQuery->execute()
    ) {
        $groupAnimalsResult = $selectGroupAnimalsQuery->get_result();
        $selectGroupAnimalsQuery->close();

        // initialize array to hold each animal (row data)
        $result = array();

        // iterate through the row results and append to results array
        while ($row = $groupAnimalsResult->fetch_assoc()) {
            $result[] = $row['animal_id'];
        }

        // make sure we have a result of group animals
        if (count($result) > 0) {
            return $result;
        } else {
            echo json_encode('No group animals result');
            exit(1);
        }
    }
}

function query($mysqli, $moveGroupQuery, $actionTypeID, $item, $amount, $oldGroupIDs, $newGroupID, $itemIDs, $userID, $allOldGroupAnimals)
{

    // if query can prepare, parameters can bind and query can execute, proceed
    if (
        $moveGroupQuery &&
        $moveGroupQuery->bind_param(
            'is',
            $newGroupID,
            $userID
        ) &&
        $moveGroupQuery->execute()
    ) {

        // grab the number of rows affected by the query
        // $rowsAffected = $mysqli->affected_rows;

        // close the query
        $moveGroupQuery->close();

        /*
        * If the user selected animal(s) which existed in multiple groups from the `Actions --> All Animals` view
            we assumed that the user wants to move the animal out of all multiple groups into the single selected (new) group
        * In doing this, this changes all instances of the animal associated with the previous (old) group(s) in
            the `group_animals` table in the database
        * If an animal was in groups with IDs 1 and 2, then the same animal has two records in the `group_animals` table.
            - If the animal is moved to a new group ID, 3, then the records in the `group_animals` table would be updated to 3
                meaning that the animal in the animal "is in the group twice".
        * Remove excess records associated with the `group_id`, `animal_id`, and `user_id`
        * NOTE: probably not the best solution, but not sure what else to do at the moment (September 16, 2020)
        */
        if ($item === 'animals') {
            foreach ($itemIDs as $id) {
                deleteExcess($mysqli, $newGroupID, $id, $userID);
            }
        }

        // Pass data to eventLog.php depending on `$item` value
        if ($item === 'animals') {
            // Single Example: ["animals","single",[[1]],7,[30]]
            // Multiple Example: ["animals","multiple",[9,9],7,[26,25]]
            $values = [$item, $amount, $oldGroupIDs, $newGroupID, $itemIDs];
        } elseif ($item === 'groups') {
            // Single Example: ["groups","single",[7],9,[[23,24,25,26,29]]]
            $values = [$item, $amount, $itemIDs, $newGroupID, $allOldGroupAnimals];
        }

        if ($values) {
            // Pass data to write to Event and Log database tables
            $result = createEvent($mysqli, $actionTypeID, $values, $userID);

            // If we have a return value of True, then we finished creating event logs for all items modified
            if ($result) {
                // echo `success` back to the app to let it know we were successful in perform Move action and logging events
                return true;
            } else {
                echo json_encode('Error creating event');
                exit(1);
            }
        } else {
            echo json_encode('No values!');
            exit(1);
        }
    } else {
        echo json_encode($mysqli->error);
        exit(1);
    }
}

// make sure we have received a post request from app
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    $data = file_get_contents('php://input');

    $request = json_decode($data);

    if ($request) {
        require('connection.php');
        // 1 == 'move'
        $actionTypeID = intval($mysqli->real_escape_string($request->action));
        // $item = 'animals' or 'groups'
        $item = $mysqli->real_escape_string($request->item);
        // $amount = 'single' or 'multiple'
        $amount = $mysqli->real_escape_string($request->amount);
        // example: [5, 7, 22]
        $oldGroupIDs = $request->oldIDs;
        // example: 99
        $newGroupID = intval($mysqli->real_escape_string($request->newID));
        // example: [[10, 12], [15], [19, 22, 27]]
        $itemIDs = $request->itemIDs;
        // example: ext-yz1
        $userID = $mysqli->real_escape_string($request->userID);
        // true
        $loggedIn = $request->loggedIn;

        if ($userID && $loggedIn) {
            if ($item === 'animals') {
                foreach ($itemIDs as $key => $value) {
                    // implode array of IDs into string for SQL query
                    $animalIDs = implode(', ', array_map('intval', $value));

                    /*
                    + If, we are dealing with animals then update the associated `group_id` 
                        of the animals in the `group_animals` table with the new group id selected by the user
                    */
                    $moveGroupQuery = $mysqli->prepare(
                        'UPDATE `group_animals` AS `ga`
                        SET `ga`.`group_id` = ?
                        WHERE `ga`.`animal_id` IN (' . $animalIDs . ')
                        AND `ga`.`user_id` = ?'
                    );

                    // Execute query for each set of animal IDs
                    $result = query($mysqli, $moveGroupQuery, $actionTypeID, $item, $amount, $oldGroupIDs[$key], $newGroupID, $value, $userID, null);

                    if (!$result) {
                        echo json_encode('Error: No Animals Result.');
                        exit(1);
                    }
                }

                echo json_encode('success');
            } elseif ($item === 'groups') {
                /*
                + Else if, we are dealing with group(s), update the associated `group_id`
                    of the group(s) to the new group id selected by the user
                */

                // initialize array to hold array of group animals for passed group ids being moved
                $allOldGroupAnimals = array();

                // iterate through passed group ids and grab all animals in group (before move)
                foreach ($itemIDs as $id) {
                    $oldGroupAnimals = getOldGroupAnimals($mysqli, $id);

                    // echo json_encode($oldGroupAnimals);

                    // if we have a array of old group animals greater than 0, append to overall list
                    if (count($oldGroupAnimals) > 0) {
                        $allOldGroupAnimals[] = $oldGroupAnimals;
                    } else {
                        // else, echo error
                        echo json_encode('Error: No old group animals.');
                        exit(1);
                    }
                }

                // if array of overall group animals has value, prepare query to move user selected groups to new group
                if (count($allOldGroupAnimals) > 0) {
                    // implode array of IDs into string for SQL query
                    $groupIDs = implode(', ', array_map('intval', $itemIDs));

                    // prepare query where `group_animals` table will be updated to replace current group id value
                    // with id of new group selected by user
                    $moveGroupQuery = $mysqli->prepare(
                        'UPDATE `group_animals` AS `ga`
                        SET `ga`.`group_id` = ?
                        WHERE `ga`.`group_id` IN (' . $groupIDs . ')
                        AND `ga`.`user_id` = ?'
                    );

                    // Execute query for group ID
                    $result = query($mysqli, $moveGroupQuery, $actionTypeID, $item, $amount, $oldGroupIDs, $newGroupID, $groupIDs, $userID, $allOldGroupAnimals);

                    if ($result) {
                        echo json_encode('success');
                    } else {
                        echo json_encode('Error: No Group Result');
                        exit(1);
                    }
                } else {
                    echo json_encode('Error: No overall old group animals.');
                    exit(1);
                }
            } else {
                echo json_encode('Item Error');
                exit(1);
            }
        } else {
            echo json_encode('Not logged in');
            exit(1);
        }

        // returns the thread id for the current connection
        $thread = $mysqli->thread_id;

        // kills the thread id for the current connection
        $mysqli->kill($thread);

        // closes the sql database connection
        $mysqli->close();
    } else {
        echo json_encode('No request data.');
        exit(1);
    }
}
