<?php
require('accessControl.php');

/*
BY:             krn
FILENAME:       addAnimals.php
DESCRIPTION:    Create new database record(s) for animal(s) in the `animals` table with
                the passed data from user input. Once the records have been created,
                the associated `id` from the `animals` table is used to create a record
                in the `group_animals` table. The `group_animals` table creates a reference
                between the group(s) and animal(s).
*/

$message = 'Add Animals Error';

// make sure we have received a post request
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    // Check that data is set and not empty
    if (isset($_POST) && !empty($_POST)) {

        require('connection.php');

        // define array to hold our form data
        $addAnimalsData = array();
        $userID = NULL;
        $loggedIn = NULL;

        // iterate through posted data as key -> value pairing
        foreach ($_POST as $key => $value) {
            // If the current `$key` is `userID` set $userID value
            if ($key === 'userID' && isset($value) && !empty($value) && $value !== 'null') {
                $userID = $value;
            }
            // Else if, the current `$key` is `loggedIn` set $loggedIn value
            elseif ($key === 'loggedIn' && isset($value) && !empty($value) && $value !== 'null') {
                $loggedIn = $value;
            }
            // if the current key is birthDate and we have a value, convert string to time, append to array
            elseif ($key === 'date') {
                if (isset($value) && !empty($value) && $value !== 'null') {
                    // Trim the value to remove whitespace from both sides
                    $trimmedValue = trim($value);
            
                    // Escape the trimmed value
                    $escapedValue = $mysqli->real_escape_string($trimmedValue);
            
                    // Convert the escaped value to date format 'Y-m-d'
                    $formattedDate = date('Y-m-d', strtotime($escapedValue));
            
                    // Append the formatted date to the list
                    $addAnimalData[] = $formattedDate;
                } else {
                    // Append null value to the list
                    $addAnimalData[] = null;
                }
            }
            // Else If value is empty or string "null", append null to array.
            elseif (empty($value) || $value === 'null') {
                $addAnimalData[] = null;
            } else {
                // Otherwise, append cleaned value
                $trimmedValue = trim($value);
                $escapedValue = $mysqli->real_escape_string($trimmedValue);
                $addAnimalData[] = $escapedValue;
            }
        }

        /*
        * Data in the `addAnimalData` array
        * [0]  = primary id (name)
        * [1]  = sex id
        * [2]  = addition type (birth [1] or purchase [2])
        * [3]  = date of birth
        * [4]  = date of purchase
        * [5]  = breed id
        * [6]  = color id
        * [7]  = group id
        * [8]  = weight
        * [9]  = tag color
        * [10]  = official NUES (national uniform eartagging system)
        * [11] = official AIN (animal identification number)
        * [12] = twin status id (only shown if user selected birth)
        * [13] = calf vigor score id (only shown if user selected birth)
        * [14] = dam (referenced animal id, selected by user), (only shown if user selected birth)
        * [15] = teat score id (only shown if user selected birth), (only shown if user selected dam)
        * [16] = udder score id (only shown if user selected birth), (only shown if user selected dam)
        * [17] = calving ease id (only shown if user selected birth), (only shown if user selected dam)
        * [18] = dam body condition score (bcs) id (only shown if user selected birth), (only shown if user selected dam)
        * [19] = sire (only shown if user selected birth), (referenced animal id, selected by user)
        * [20] = status; user can select if the animal is "Active" (1) or "Inactive" (2); default value (1)
        * [21] = reason; if the user selects "Inactive" (2) for the animal status, ask for a reason (and cause)
        * [22] = cause; if the user selects "Inactive" (2) for the animal status, ask for cause (and reason)
        * [23] = notes; optional field for the user to enter any text related to animal notes they may want to record
        */

        if ($userID && $loggedIn) {

            // set group id from the animal data, define as integer, to be used later
            $groupID = (int) $addAnimalData[7];

            // NOTE: Functionality related to record types has not been implemented (AS OF JAN. 2023)
            // INITIALIZE AS DEFAULT; STATIC FOR TESTING REMOVE BEFORE PRODUCTION
            // record type (1 = full access)
            $recordTypeID = 1;

            // define local time zone and format
            date_default_timezone_set('America/Chicago');
            // set date created/updated for record insertion
            $dateCreated = date('Y-m-d H:i:s');
            $dateUpdated = date('Y-m-d H:i:s');

            // initialize query
            $addAnimalQuery = $mysqli->prepare(
                'INSERT INTO `animals`
                (`primary_id`, `sex_id`, `addition_type`, `date_birth`, `date_purchase`,
                `breed_id`, `color_id`, `weight`, `tag_color`,
                `official_nues`, `official_ain`, `twin_status_id`,  `calving_vigor_score_id`,
                `dam`, `teat_score_id`, `udder_score_id`, `calving_ease_id`,
                `dam_bcs_id`,  `sire`, `status_id`, `reason_id`, `cause_id`, `notes`,
                `user_id`, `record_type_id`, `date_created`, `date_updated`)
                VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)'
            );

            // if the query can prepare, bind parameters, and execute, then continue
            if (
                $addAnimalQuery &&
                $addAnimalQuery->bind_param(
                    'siissiidsssiiiiiiiiiiississ',
                    $addAnimalData[0],  // primary id (animal name)
                    $addAnimalData[1],  // sex id
                    $addAnimalData[2],  // addition type (birth [1] or purchase [2])
                    $addAnimalData[3],  // date of birth
                    $addAnimalData[4],  // date of purchase
                    $addAnimalData[5],  // breed id
                    $addAnimalData[6],  // color id
                    // $addAnimalData[7] // GROUP ID IS NOT ADDED TO ANIMAL DATA SO SKIP; SEE $groupID ABOVE
                    $addAnimalData[8],  // weight (lbs)
                    $addAnimalData[9],  // tag color
                    $addAnimalData[10],  // official NUES (national uniform eartagging system)
                    $addAnimalData[11], // official AIN (animal identification number)
                    $addAnimalData[12], // twin status id
                    $addAnimalData[13], // calving vigor score id
                    $addAnimalData[14], // dam id (referenced animal id, selected by user)
                    $addAnimalData[15], // teat score id
                    $addAnimalData[16], // udder score id
                    $addAnimalData[17], // calving ease id
                    $addAnimalData[18], // dam body condition score id
                    $addAnimalData[19], // sire id (referenced animal id, selected by user)
                    $addAnimalData[20], // status id (active/inactive)
                    $addAnimalData[21], // reason id (related to inactive status)
                    $addAnimalData[22], // cause id (related to reason of inactive status)
                    $addAnimalData[23], // notes
                    $userID,            // id of the user adding animals
                    $recordTypeID,      // type of record for animals being added
                    $dateCreated,       // set date created for animal record
                    $dateUpdated        // set date updated for animal record
                ) &&
                $addAnimalQuery->execute()
            ) {
                // close query to add animal
                $addAnimalQuery->close();

                // grab database record id for new animal
                $animalID = $mysqli->insert_id;

                /*
                * If we have a value from the database with the insert id for the animal
                    that was added, and we have a group ID, and the status for the animal is "Active" (1),
                    then proceed to insert a record into the `group_animals` table
                    that links the animal to the group in which it was added in (via the app)
                */
                if ($animalID && $groupID) {

                    // initialize query
                    $addGroupAnimalsQuery = $mysqli->prepare(
                        'INSERT INTO `group_animals`
                            (`group_id`, `animal_id`, `user_id`, `entry_date`)
                            VALUES (?,?,?,?)'
                    );

                    // if the query can prepare, bind parameters, and execute, then continue
                    if (
                        $addGroupAnimalsQuery &&
                        $addGroupAnimalsQuery->bind_param(
                            'iiss',
                            $groupID,
                            $animalID,
                            $userID,
                            $dateCreated
                        ) &&
                        $addGroupAnimalsQuery->execute()
                    ) {
                        $message = 'success';
                    } else {
                        $message = $mysqli->error;
                    }

                    $addGroupAnimalsQuery->close();
                } else {
                    $message = 'Error associating animal with group!';
                }
            } else {
                $message = $mysqli->error;
            }
        } else {
            $message = 'User is not logged in!';
        }
    } else {
        $message = 'No add animal data!';
    }
}

echo json_encode($message);
$mysqli->close();
