<?php
session_start();
require('accessControl.php');

$userID = null;
$loggedIn = null;

/*
BY:             krn
FILENAME:       getAllAnimals.php
DESCRIPTION:    Get all animal records from the database for the current user.
                Once animal records are retrieved, query for additional dam and sire
                data linked to the animal record.
*/

// Handle OPTIONS requests separately
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    // Pre-flight request. Reply successfully:
    http_response_code(200);
    exit;
}

// Check if the request method is POST, then the user is accessing via the app
// If so, handle the POST data as JSON
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Receive JSON POST from app
    $data = file_get_contents('php://input');

    // Check that data is set and not empty
    if (isset($data) && !empty($data)) {
        // Decode JSON POST data
        $info = json_decode($data);

        // Check that object is set and not empty
        if (isset($info) && !empty($info)) {
            // Define variables
            $userID = $info->userID;
            $loggedIn = $info->loggedIn;
        }
    }
} elseif (isset($_SESSION['authenticated']) && $_SESSION['authenticated'] && isset($_SESSION['userID'])) {
    // If the request method is not POST, then the user is accessing via the portal
    // Check $_SESSION variables to ensure the user is logged in
    $userID = $_SESSION['userID'];
    $loggedIn = true;
} else {
    // If the user is not logged in, redirect to home page
    header('Location: ../portal/logout.php');
}

// If the user has been authenticated (logged in)
if ($userID && $loggedIn) {
    // Require database connection
    require('connection.php');

    $selectAllAnimalsQuery = $mysqli->prepare(
        'SELECT
            `a`.*,
            `s`.`name` AS `sex_name`,
            `sr`.`description` AS `inactive_reason`,
            `sc`.`description` AS `inactive_cause`,
            GROUP_CONCAT(DISTINCT `g`.`group_name`) AS `group_names`,
            GROUP_CONCAT(DISTINCT `ga`.`group_id`) AS `group_ids`,
            GROUP_CONCAT(DISTINCT `gc`.`category_name`) AS `category_names`,
            GROUP_CONCAT(DISTINCT `gc`.`id`) AS `category_ids`
        FROM
            `animals` AS `a`
            JOIN `group_animals` AS `ga` ON `a`.id = `ga`.`animal_id`
            JOIN `groups` AS `g` ON `ga`.`group_id` = `g`.`id`
            JOIN `group_categories` AS `gc` ON `g`.`category_id` = `gc`.`id`
            JOIN `sexes` AS `s` ON `a`.`sex_id` = `s`.`id`
            JOIN `status_reason` AS `sr` ON `a`.`reason_id` = `sr`.`id`
            JOIN `status_cause` AS `sc` ON `a`.`cause_id` = `sc`.`id`
        WHERE `a`.`user_id` = ?
        GROUP BY `a`.`id`
        ORDER BY `a`.`primary_id` ASC'
    );

    /*
    * If query was successful, then bind parameters, and execute query
    * Get all group associated with the current user (`$userID`)
    * If the query can be prepared, parameters can be bind, and query can execute, continue with results
    */
    if (
        $selectAllAnimalsQuery &&
        $selectAllAnimalsQuery->bind_param('s', $userID) &&
        $selectAllAnimalsQuery->execute()
    ) {
        // Get result from query and close query
        $animalResult = $selectAllAnimalsQuery->get_result();
        $selectAllAnimalsQuery->close();

        // Initialize array to hold each group (row data)
        $result = array();

        // Iterate through results by each row and append to result array
        while ($row = $animalResult->fetch_assoc()) {
            $result[] = $row;
        }

        /*
        * If there is a result
        * Iterate through each result and append dam or sire name
            if animal ID for either column is available
        */
        if ($result) {
            $newResult = array();

            // Prepare the statements for dam and sire name queries before the loop
            $selectDamNameQuery = $mysqli->prepare(
                'SELECT `a`.`primary_id` AS `dam_name`
                FROM `animals` AS `a`
                WHERE `a`.`id` = ?
                AND `a`.`user_id` = ?'
            );

            $selectSireNameQuery = $mysqli->prepare(
                'SELECT `a`.`primary_id` AS `sire_name`
                FROM `animals` AS `a`
                WHERE `a`.`id` = ?
                AND `a`.`user_id` = ?'
            );

            foreach ($result as $r) {
                if ($r['dam']) {
                    if (
                        $selectDamNameQuery &&
                        $selectDamNameQuery->bind_param('is', $r['dam'], $userID) &&
                        $selectDamNameQuery->execute()
                    ) {
                        // Grab result from query and close query
                        $damNameResult = $selectDamNameQuery->get_result();
                        $damName = $damNameResult->fetch_assoc();

                        $r['dam_name'] = $damName['dam_name'] ?? null;
                    } else {
                        $r['dam_name'] = null;
                    }
                } else {
                    $r['dam_name'] = null;
                }

                if ($r['sire']) {
                    if (
                        $selectSireNameQuery &&
                        $selectSireNameQuery->bind_param('is', $r['sire'], $userID) &&
                        $selectSireNameQuery->execute()
                    ) {
                        // Grab result from query and close query
                        $sireNameResult = $selectSireNameQuery->get_result();
                        $sireName = $sireNameResult->fetch_assoc();

                        $r['sire_name'] = $sireName['sire_name'] ?? null;
                    } else {
                        $r['sire_name'] = null;
                    }
                } else {
                    $r['sire_name'] = null;
                }

                $newResult[] = $r;
            }

            // Close the dam and sire name queries after the loop
            $selectDamNameQuery->close();
            $selectSireNameQuery->close();

            // Echo (return) array of groups for the current user
            // echo json_encode(['status' => 'success', 'data' => $newResult]);
            echo json_encode($newResult);
        } else {
            // echo json_encode(['status' => 'error', 'data' => 'No results!']);
            echo json_encode('Error: No results!');
        }
    } else {
        /*
        * Else, query failed to prepare, bind or execute
        * Echo (return) mysqli error
        */
        // echo json_encode(['status' => 'error', 'data' => $mysqli->error]);
        echo json_encode($mysqli->error);
    }

    // Close database connection
    $mysqli->close();
} else {
    // echo json_encode(['status' => 'error', 'data' => 'User ID is not provided or empty']);
    echo json_encode('Error: User ID is not provided or empty');
}
