<?php
session_start();
require('accessControl.php');

$userID = null;
$loggedIn = null;

/*
BY:             krn
FILE:           getGroups.php
DESCRIPTION:    Get all group data, associated group category name and counts the amount
				of group IDs for each group animal pair (number of amounts in a specific group)
				by joining the associated `group_categories` and `group_animal` tables
				with the `groups` table for the current user, based on the passed `userID`.
				The resulting data is returned and ordered by the date a group was last updated in descending order.
*/

// Handle OPTIONS requests separately
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
	// Pre-flight request. Reply successfully:
	http_response_code(200);
	exit;
}

// Check if the request method is POST, then the user is accessing via the app
// If so, handle the POST data as JSON
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
	// Receive JSON POST from app
	$data = file_get_contents('php://input');

	// Check that data is set and not empty
	if (isset($data) && !empty($data)) {
		// Decode JSON POST data
		$info = json_decode($data);

		// Check that object is set and not empty
		if (isset($info) && !empty($info)) {
			// Define variables
			$userID = $info->userID;
			$loggedIn = $info->loggedIn;
		}
	}
} elseif (isset($_SESSION['authenticated']) && $_SESSION['authenticated'] && isset($_SESSION['userID'])) {
	// If the request method is not POST, then the user is accessing via the portal
	// Check $_SESSION variables to ensure the user is logged in
	$userID = $_SESSION['userID'];
	$loggedIn = true;
} else {
	// If the user is not logged in, redirect to home page
	header('Location: ../portal/logout.php');
}


// If the user has been authenticated (logged in)
if ($userID && $loggedIn) {
	// Require database connection
	require('connection.php');

	// Prepare `select groups` query
	$selectGroupsQuery = $mysqli->prepare(
		'SELECT `g`.*, `gc`.`category_name`,
		COUNT(`ga`.`group_id`) AS `amount_animals`
		FROM `groups` AS `g`
		JOIN `group_categories` AS `gc`
		ON `g`.`category_id` = `gc`.`id`
		LEFT JOIN `group_animals` AS `ga`
		ON `g`.`id` = `ga`.`group_id`
		WHERE `g`.`user_id` = ?
		GROUP BY `g`.`id`
		ORDER BY `g`.`group_name` ASC'
	);

	/*
	* Get all group associated with the current user (`$userID`)
	* If the query can be prepared, parameters can be bind, and query can execute, continue with results
	*/
	if (
		$selectGroupsQuery &&
		$selectGroupsQuery->bind_param('s', $userID) &&
		$selectGroupsQuery->execute()
	) {

		// Get result from query and close query
		$groupResult = $selectGroupsQuery->get_result();

		// Initialize array to hold each group (row data)
		$result = array();

		// Iterate through results by each row and append to result array
		while ($row = $groupResult->fetch_assoc()) {
			$result[] = $row;
		}

		$selectGroupsQuery->close();

		// Echo (return) array of groups for the current user
		echo json_encode($result);
	} else {
		/*
        * Else, query failed to prepare, bind or execute
        * Echo (return) mysqli error
        */
		// echo json_encode(['status' => 'error', 'data' => $mysqli->error]);
		echo json_encode($mysqli->error);
	}

	// Close database connection
	$mysqli->close();
} else {
	// echo json_encode(['status' => 'error', 'data' => 'User ID is not provided or empty']);
	echo json_encode('Error: User ID is not provided or empty');
}
