<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
require('accessControl.php');

/*
BY:             krn
FILENAME:       reports.php
DESCRIPTION:    Query the database and generate reports based on user's request. Corresponds to reports
                present in the CALF manager app.
*/

$message = 'Error: General error with reports!';

function breedingReportQuery($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID, $reportType)
{
    try {
        $beID = $reportType === 'air' ? 3 : ($reportType === 'nsbr' ? 4 : 0);

        if ($beID !== 3 && $beID !== 4) {
            echo json_encode('Invalid Report Type!: ' . $reportType);
            return;
        }

        $baseQuery = "SELECT
            COW.`id` AS 'Animal_ID',
            COW.`primary_id` as 'Cow',
            G.`group_name` AS 'Cow_Group',
            BULL.`primary_id` AS 'Bull',
            G1.`group_name` AS 'Bull_Group',
            EB.`date` AS 'Date',
            DATE_FORMAT(EB.`date` + INTERVAL 283 DAY, '%Y-%m-%d') AS 'Expected_Calving_Date',
            EB.`technician` AS 'Technician',
            EB.`notes` AS 'Notes',
            EB.`event_datetime` AS 'Date_Recorded'
            FROM animals AS COW
            JOIN event_breeding AS EB ON EB.`animal_id` = COW.`id`
            JOIN breeding_event_types AS BET ON BET.`id` = EB.`event_type_id`
            JOIN animals AS BULL ON BULL.`id` = EB.`bull_id`
            LEFT JOIN `group_animals` AS COW_GROUP ON COW_GROUP.`animal_id` = COW.`id`
            LEFT JOIN `groups` AS G ON G.`id` = COW_GROUP.`group_id`
            LEFT JOIN `group_animals` AS BULL_GROUP ON BULL_GROUP.`animal_id` = BULL.`id`
            LEFT JOIN `groups` AS G1 ON G1.`id` = BULL_GROUP.`group_id`";

        // Initialize ID and Date conditions
        $idCondition = '';
        $dateCondition = '';

        // Check the ID type and build the corresponding condition
        if ($idType == 'groups' && !empty($IDs)) {
            $idPlaceholders = implode(',', array_fill(0, count($IDs), '?'));
            $idCondition = " AND COW_GROUP.`group_id` IN ($idPlaceholders)";
        } elseif ($idType == 'animals' && !empty($IDs)) {
            $idPlaceholders = implode(',', array_fill(0, count($IDs), '?'));
            $idCondition = " AND COW.`id` IN ($idPlaceholders)";
        }

        // Initialize dateCondition to handle both start and end dates independently
        if ($startDate && $endDate) {
            $dateCondition .= " AND EB.`date` BETWEEN ? AND ?";
        } elseif ($startDate) {
            $dateCondition .= " AND EB.`date` >= ?";
        } elseif ($endDate) {
            $dateCondition .= " AND EB.`date` <= ?";
        }

        // Process statusID to handle multiple values
        $statusIDs = explode(',', $statusID);
        $statusPlaceholders = implode(',', array_fill(0, count($statusIDs), '?'));

        // Complete the query using the constructed conditions
        $baseQuery .= " WHERE EB.`event_type_id` = ?
        AND COW.`user_id` = ?
        AND COW.`status_id` IN ($statusPlaceholders)
        AND COW.`sex_id` = 3
        AND COW.`addition_type` = 1
        $idCondition
        $dateCondition
        ORDER BY EB.`date` DESC";

        // Prepare the SQL query
        $query = $mysqli->prepare($baseQuery);
        if ($query === false) {
            die('Prepare Error: ' . $mysqli->error);
        }

        // Initialize parameters and their types
        $params = [$beID, $userID];
        $paramTypes = 'is';

        // Append statusIDs to parameters
        foreach ($statusIDs as $sid) {
            // Casting to integer for safety
            $params[] = (int)$sid;
            $paramTypes .= 'i';
        }

        // Append group/animal IDs to parameters if provided
        if (!empty($IDs)) {
            foreach ($IDs as $id) {
                // Casting to integer for safety
                $params[] = (int)$id;
                $paramTypes .= 'i';
            }
        }

        // Append dates to parameters if provided
        if ($startDate && $endDate) {
            $params[] = $startDate;
            $params[] = $endDate;
            $paramTypes .= 'ss';
        } elseif ($startDate) {
            $params[] = $startDate;
            $paramTypes .= 's';
        } elseif ($endDate) {
            $params[] = $endDate;
            $paramTypes .= 's';
        }

        // Bind the parameters using the constructed values and types
        $query->bind_param($paramTypes, ...$params);

        // Execute the query and fetch results
        if ($query->execute()) {
            $breedingResult = $query->get_result();
            $query->close();

            // Collect and encode the result as JSON
            $result = array();
            while ($row = $breedingResult->fetch_assoc()) {
                $result[] = $row;
            }

            echo json_encode($result);
        } else {
            echo json_encode($mysqli->error);
            exit(1);
        }
    } catch (Exception $e) {
        echo 'Caught exception on breedingReportQuery(): ' . $e->getMessage() . ' for report type: ' . $reportType . '\n';
        exit(1);
    }
}


function bullInBullOutQuery($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID, $reportType)
{
    try {
        // Check if the report type is valid for this function
        if ($reportType !== 'bior') {
            echo json_encode('Invalid Bull-In/Bull-Out (BIOR) Report Type!');
            return;
        }

        // Base query with subqueries for Bull-In and Bull-Out events
        $baseQuery = "SELECT
            T1.`Animal_ID` AS 'Animal_ID',
            T1.`cow` AS 'Cow',
            T1.`bull` AS 'Bull',
            T1.`Bull_In_Date` AS 'Bull_In_Date',
            DATE_FORMAT(DATE_ADD(T1.`Bull_In_Date`, INTERVAL 283 DAY), '%Y-%m-%d') AS 'Earliest_Calving_Date',
            T1.`Bull_In_Technician` AS 'Bull_In_Technician',
            T1.`Bull_In_Notes` AS 'Bull_In_Notes',
            T2.`Bull_Out_Date` AS 'Bull_Out_Date',
            DATE_FORMAT(DATE_ADD(T2.`Bull_Out_Date`, INTERVAL 283 DAY), '%Y-%m-%d') AS 'Latest_Calving_Date',
            T2.`Bull_Out_Technician` AS 'Bull_Out_Technician',
            T2.`Bull_Out_Notes` AS 'Bull_Out_Notes'
        FROM `group_animals` AS GA
        JOIN (
            SELECT
                A.`id` AS 'Animal_ID',
                A.`primary_id` AS 'cow',
                A.`status_id` AS 'Status_ID',
                EB.`date` AS 'Bull_In_Date',
                A1.`primary_id` AS 'bull',
                EB.`technician` AS 'Bull_In_Technician',
                EB.`notes` AS 'Bull_In_Notes'
            FROM animals AS A
            JOIN event_breeding AS EB ON EB.`animal_id` = A.`id`
            JOIN breeding_event_types AS BET ON BET.`id` = EB.`event_type_id`
            JOIN animals AS A1 ON A1.`id` = EB.`bull_id`
            WHERE A.`sex_id` = 3 AND EB.`event_type_id` = 1
        ) AS T1 ON GA.`animal_id` = T1.`Animal_ID`
        LEFT JOIN (
            SELECT
                EB.`animal_id` AS 'Animal_ID',
                EB.`date` AS 'Bull_Out_Date',
                EB.`technician` AS 'Bull_Out_Technician',
                EB.`notes` AS 'Bull_Out_Notes'
            FROM event_breeding AS EB
            JOIN breeding_event_types AS BET ON BET.`id` = EB.`event_type_id`
            WHERE EB.`event_type_id` = 2
        ) AS T2 ON GA.`animal_id` = T2.`Animal_ID`";

        // Initialize ID and Date conditions
        $idCondition = '';
        $dateCondition = '';

        // Check the ID type and build the corresponding condition
        if ($idType == 'groups' && !empty($IDs)) {
            $idPlaceholders = implode(',', array_fill(0, count($IDs), '?'));
            $idCondition = " AND GA.`group_id` IN ($idPlaceholders)";
        } elseif ($idType == 'animals' && !empty($IDs)) {
            $idPlaceholders = implode(',', array_fill(0, count($IDs), '?'));
            $idCondition = " AND A.`id` IN ($idPlaceholders)";
        }

        // Building date condition based on startDate and endDate
        if ($startDate && $endDate) {
            $dateCondition .= " AND T1.`Bull_In_Date` BETWEEN ? AND ?";
        } elseif ($startDate) {
            $dateCondition .= " AND T1.`Bull_In_Date` >= ?";
        } elseif ($endDate) {
            $dateCondition .= " AND T1.`Bull_In_Date` <= ?";
        }

        // Process statusID to handle multiple values
        $statusIDs = explode(',', $statusID);
        $statusPlaceholders = implode(',', array_fill(0, count($statusIDs), '?'));

        // Appending conditions to the base query
        $baseQuery .= " WHERE GA.`user_id` = ?
        AND T1.`Status_ID` IN ($statusPlaceholders)
        $idCondition
        $dateCondition
        ";

        // Preparing the SQL query
        $query = $mysqli->prepare($baseQuery);
        if (!$query) {
            echo json_encode($mysqli->error);
            return;
        }

        // Initialize parameters and their types
        $params = [$userID];
        $paramTypes = 's';

        // Append statusIDs to parameters
        foreach ($statusIDs as $sid) {
            // Casting to integer for safety
            $params[] = (int)$sid;
            $paramTypes .= 'i';
        }

        // Append IDs to parameters if provided
        if (!empty($IDs)) {
            foreach ($IDs as $id) {
                // Casting to integer for safety
                $params[] = (int)$id;
                $paramTypes .= 'i';
            }
        }

        // Append dates to parameters if provided
        if ($startDate && $endDate) {
            $params[] = $startDate;
            $params[] = $endDate;
            $paramTypes .= 'ss';
        } elseif ($startDate) {
            $params[] = $startDate;
            $paramTypes .= 's';
        } elseif ($endDate) {
            $params[] = $endDate;
            $paramTypes .= 's';
        }

        // Binding the parameters and executing the query
        if (!$query->bind_param($paramTypes, ...$params) || !$query->execute()) {
            echo json_encode($mysqli->error);
            return;
        }

        // Retrieving the data from the query
        $data = $query->get_result();
        $query->close();

        // Collecting and encoding the results as JSON
        $results = array();
        while ($row = $data->fetch_assoc()) {
            $results[] = $row;
        }

        echo json_encode($results);
    } catch (Exception $e) {
        echo 'Caught exception on bullInBullOutQuery(): ' . $e->getMessage() . ' for report type: ' . $reportType . '\n';
        exit(1);
    }
}


function cowCalvingReportQuery($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID)
{
    try {
        // Base query for fetching cow calving report data
        $baseQuery = "SELECT
            A1.`id` AS 'Dam_ID',
            A1.`primary_id` AS 'Dam',
            A.`id` AS 'Calf_ID',
            A.`primary_id` AS 'Calf',
            A.`date_birth` AS 'Calf_Birth_Date',
            A.`weight` AS 'Calf_Birth_Weight',
            B.`name` AS 'Breed',
            C.`title` as 'Color',
            A.`tag_color` AS 'Tag_Color',
            TWIN.`description` AS 'Twin Status',
            CVS.`title` AS 'Calf Vigor',
            A2.`primary_id` AS 'Sire',
            A.`teat_score_id` AS 'Teat Score',
            A.`udder_score_id` AS 'Udder Score',
            CE.`description` AS 'Calving Ease',
            A.`dam_bcs_id` AS 'Dam_BCS'
            FROM animals AS A
            LEFT JOIN animals AS A1 ON A1.`id` = A.`dam`
            LEFT JOIN breeds as B ON B.`id` = A.`breed_id`
            LEFT JOIN colors as C ON C.`id` = A.`color_id`
            LEFT JOIN twin_status AS TWIN ON TWIN.`id` = A.`twin_status_id`
            LEFT JOIN calving_vigor_score AS CVS ON CVS.`id` = A.`calving_vigor_score_id`
            LEFT JOIN calving_ease AS CE ON CE.`id` = A.`calving_ease_id`
            LEFT JOIN animals AS A2 ON A2.`id` = A.`sire`";

        // Check the ID type and build the corresponding condition
        if ($idType == 'groups' && !empty($IDs)) {
            // For groups, join with group_animals and groups tables
            $baseQuery .= " JOIN `group_animals` AS GA ON GA.`animal_id` = A.`id`
                            JOIN `groups` AS G ON G.`id` = GA.`group_id`";
            $idPlaceholders = implode(',', array_fill(0, count($IDs), '?'));
            $idCondition = " GA.`group_id` IN ($idPlaceholders)";
        } elseif (!empty($IDs)) {
            // For individual animals, use their IDs directly
            $idPlaceholders = implode(',', array_fill(0, count($IDs), '?'));
            $idCondition = " A.`id` IN ($idPlaceholders)";
        } else {
            // If no IDs provided, this condition will be ignored
            $idCondition = "1";
        }

        // Initialize dateQuery to handle both start and end dates independently
        $dateQuery = '';
        if ($startDate && $endDate) {
            // Both start and end dates are provided
            $dateQuery .= " AND A.`date_birth` BETWEEN ? AND ?";
        } elseif ($startDate) {
            // Only start date is provided
            $dateQuery .= " AND A.`date_birth` >= ?";
        } elseif ($endDate) {
            // Only end date is provided
            $dateQuery .= " AND A.`date_birth` <= ?";
        }

        // Complete the query using the constructed conditions
        $baseQuery .= " JOIN `statuses` AS ST ON ST.`id` = A.`status_id`
        WHERE A.`status_id` IN (" . $statusID . ")
        AND A.`addition_type` = 1
        AND A.`user_id` = ?
        AND " . $idCondition . $dateQuery;

        // Prepare the SQL query
        $query = $mysqli->prepare($baseQuery);
        if ($query === false) {
            die('Prepare Error: ' . $mysqli->error);
        }

        // Initialize parameters and their types
        $params = [$userID];
        $paramTypes = 's';

        // Append IDs to parameters if provided
        if (!empty($IDs)) {
            $params = array_merge($params, $IDs);
            $paramTypes .= str_repeat('i', count($IDs));
        }

        // Append dates to parameters if provided
        if ($startDate && $endDate) {
            $params[] = $startDate;
            $params[] = $endDate;
            $paramTypes .= 'ss';
        } elseif ($startDate) {
            $params[] = $startDate;
            $paramTypes .= 's';
        } elseif ($endDate) {
            $params[] = $endDate;
            $paramTypes .= 's';
        }

        // Bind the parameters using the constructed values and types
        $query->bind_param($paramTypes, ...$params);

        // Execute the query and fetch results
        if ($query->execute()) {
            $cowCalvingResult = $query->get_result();
            $query->close();

            // Collect and encode the result as JSON
            $result = array();
            while ($row = $cowCalvingResult->fetch_assoc()) {
                $result[] = $row;
            }

            echo json_encode($result);
        } else {
            echo json_encode($mysqli->error);
            exit(1);
        }
    } catch (Exception $e) {
        // Handle exceptions and print error message
        echo 'Caught exception on cowCalvingReportQuery():', $e->getMessage(), '\n';
        exit(1);
    }
}



function weaningReportQuery($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID)
{
    try {
        // $weaningQuery = "SELECT
        //     A.`primary_id` AS 'Animal',
        //     A1.`primary_id` AS 'Dam',
        //     EC.`event_datetime` AS 'Weaning_Date',
        //     EC.`weaning_weight` AS 'Weaning_Weight',
        //     A1.`weight` AS 'Dam_Weight',
        //     A.`dam_bcs_id` AS 'Dam_BCS',
        //     A.`weight` AS 'Calf_Birth_Weight',
        //     A.`date_birth` AS 'Calf_Birth_Date'
        //     FROM `animals` AS A
        //     JOIN `animals` AS A1
        //     ON A1.`id` = A.`dam`
        //     JOIN `group_animals` AS GA
        //     ON GA.`animal_id` = A.`id`
        //     JOIN `groups` AS G
        //     ON G.`id` = GA.`group_id`
        //     JOIN `statuses` AS ST
        //     ON ST.`id` = A.`status_id`
        //     JOIN `event_custom` AS EC
        //     ON EC.`animal_id` = A.`id`
        //     WHERE A.`status_id` IN (" . $statusID . ")
        //     AND A.`user_id` = ?";

        $weaningQuery = "SELECT
        A.`primary_id` AS 'Animal',
        A1.`primary_id` AS 'Dam',
        EC.`event_datetime` AS 'Date',
        EC.`weight` AS 'Weight',
        A1.`weight` AS 'Dam_Weight',
        A.`dam_bcs_id` AS 'Dam_BCS',
        A.`weight` AS 'Calf_Birth_Weight',
        A.`date_birth` AS 'Calf_Birth_Date'
        FROM `animals` AS A
        LEFT JOIN `animals` AS A1 ON A1.`id` = A.`dam`
        JOIN `group_animals` AS GA ON GA.`animal_id` = A.`id`
        JOIN `groups` AS G ON G.`id` = GA.`group_id`
        JOIN `statuses` AS ST ON ST.`id` = A.`status_id`
        JOIN `event_custom` AS EC ON EC.`animal_id` = A.`id`
        JOIN `weight_types` AS WT ON EC.`weight_id` = WT.`id`
        WHERE A.`status_id` IN (" . $statusID . ")
        AND A.`user_id` = ?
        AND WT.`id` = 1";

        // Initializing the parameters
        $params = [$userID];
        $paramTypes = 's';

        // Appending to WHERE clause for groups or animals
        if (($idType == 'groups' || $idType == 'animals') && !empty($IDs)) {
            $idPlaceholders = implode(',', array_fill(0, count($IDs), '?'));
            $weaningQuery .= ($idType == 'groups') ? " AND G.`id` IN ($idPlaceholders)" : " AND A.`id` IN ($idPlaceholders)";
            $params = array_merge($params, $IDs);
            $paramTypes .= str_repeat('i', count($IDs)); // 'i' for integers
        }

        // Appending to WHERE clause if start date is set
        if ($startDate) {
            $weaningQuery .= " AND EC.`event_datetime` >= ?";
            $params[] = $startDate;
            $paramTypes .= 's';
        }

        // Appending to WHERE clause if end date is set
        if ($endDate) {
            $weaningQuery .= " AND EC.`event_datetime` < ?";
            $params[] = $endDate;
            $paramTypes .= 's';
        }

        // $weaningQuery .= " AND EC.`weaning_weight` IS NOT NULL ORDER BY EC.`event_datetime` DESC";
        $weaningQuery .= " AND EC.`weight` IS NOT NULL ORDER BY EC.`event_datetime` DESC";

        // Preparing the query
        $query = $mysqli->prepare($weaningQuery);

        // Binding parameters
        $query->bind_param($paramTypes, ...$params);

        // Executing the query and fetching the result
        if ($query->execute()) {
            $weaningResult = $query->get_result();
            $query->close();

            $result = array();
            while ($row = $weaningResult->fetch_assoc()) {
                $result[] = $row;
            }

            echo json_encode($result);
        } else {
            echo json_encode($mysqli->error);
            exit(1);
        }
    } catch (Exception $e) {
        echo 'Caught exception on weaningReportQuery():', $e->getMessage(), '\n';
        exit(1);
    }
}


class Health
{
    public $health_type;
    public $id;
    public $animal_id;
    public $date;
    public $weight;
    public $product;
    public $other;
    public $system;
    public $condition;
    public $unit;
    public $route;
    public $reason;
    public $volume;
    public $serial;
    public $expiration;
    public $notes;
}

function prepareCustomHealthQuery($mysqli, $healthIDs, $userID)
{
    try {
        $healthQuery = $mysqli->prepare(
            "SELECT
        BS.`system_name` AS 'system',
        BSC.`condition_name` AS 'condition',
        EH.`product` AS 'product',
        EH.`product_other` AS 'product_other',
        EH.`volume` AS 'volume',
        U.`description` AS 'unit',
        ROA.`description` AS 'route',
        TR.`name` AS 'reason',
        EH.`weight` AS 'weight',
        EH.`notes` AS 'notes'
        FROM `event_health` AS EH
        JOIN body_systems AS BS
        ON BS.`id` = EH.`problem_bs`
        JOIN body_system_conditions AS BSC
        ON BSC.`id` = EH.`problem_bsc`
        LEFT JOIN units AS U
        ON U.`id` = EH.`units_id`
        JOIN route_of_administration AS ROA
        ON ROA.`id` = EH.`roa_id`
        JOIN treatment_reasons AS TR
        ON TR.`id` = EH.`reason`
        WHERE EH.`id` IN (" . implode(",", array_fill(0, count($healthIDs), "?")) . ")
        AND EH.`user_id` = ?"
        );

        $healthQueryParams = array_merge($healthIDs, array($userID));
        $healthQueryParamsTypes = str_repeat("i", count($healthIDs)) . "s";

        if (!$healthQuery->bind_param($healthQueryParamsTypes, ...$healthQueryParams)) {
            die("Failed to bind health parameters: (" . $healthQuery->errno . ") " . $healthQuery->error);
        }

        if (!$healthQuery->execute()) {
            die("Failed to execute health query: (" . $healthQuery->errno . ") " . $healthQuery->error);
        }

        return $healthQuery;
    } catch (Exception $e) {
        echo 'Caught exception on prepareCustomHealthQuery():', $e->getMessage(), '\n';
        exit(1);
    }
}

function assignCustomHealthData($current_data, $healthRow, $health_id)
{
    try {
        $current_data->health_type = 'treatment';
        $current_data->id = $health_id;
        $current_data->system = $healthRow['system'];
        $current_data->condition = $healthRow['condition'];
        $current_data->product = $healthRow['product'];
        $current_data->other = $healthRow['product_other'];
        $current_data->volume = $healthRow['volume'];
        $current_data->unit = $healthRow['unit'];
        $current_data->route = $healthRow['route'];
        $current_data->reason = $healthRow['reason'];
        $current_data->notes = $healthRow['notes'];

        return $current_data;
    } catch (Exception $e) {
        echo 'Caught exception on assignCustomHealthData():', $e->getMessage(), '\n';
        exit(1);
    }
}

function prepareCustomVaccinationQuery($mysqli, $vaccinationIDs, $userID)
{
    try {
        $vaccinationQuery = $mysqli->prepare(
            "SELECT
        EV.`vaccine` AS 'vaccine',
        EV.`vaccine_other` AS 'vaccine_other',
        EV.`volume` AS 'volume',
        U.`description` AS 'unit',
        ROA.`description` AS 'route',
        EV.`serial_number` AS 'serial_number',
        EV.`expiration_date` AS 'expiration_date'
        FROM `event_vaccination` AS EV
        JOIN units AS U
        ON U.`id` = EV.`units_id`
        JOIN route_of_administration AS ROA
        ON ROA.`id` = EV.`roa_id`
        WHERE EV.`id` IN (" . implode(",", array_fill(0, count($vaccinationIDs), "?")) . ")
        AND EV.`user_id` = ?"
        );

        $vaccinationQueryParams = array_merge($vaccinationIDs, array($userID));
        $vaccinationQueryParamsTypes = str_repeat("i", count($vaccinationIDs)) . "s";

        if (!$vaccinationQuery->bind_param($vaccinationQueryParamsTypes, ...$vaccinationQueryParams)) {
            die("Failed to bind parameters: (" . $vaccinationQuery->errno . ") " . $vaccinationQuery->error);
        }

        if (!$vaccinationQuery->execute()) {
            die("Failed to execute query: (" . $vaccinationQuery->errno . ") " . $vaccinationQuery->error);
        }

        return $vaccinationQuery;
    } catch (Exception $e) {
        echo 'Caught exception on prepareCustomVaccinationQuery():', $e->getMessage(), '\n';
        exit(1);
    }
}

function assignCustomVaccinationData($current_data, $vaccinationRow, $vaccination_id)
{
    try {
        $current_data->health_type = 'vaccination';
        $current_data->id = $vaccination_id;
        $current_data->product = $vaccinationRow['vaccine'];
        $current_data->other = $vaccinationRow['vaccine_other'];
        $current_data->volume = $vaccinationRow['volume'];
        $current_data->unit = $vaccinationRow['unit'];
        $current_data->route = $vaccinationRow['route'];
        $current_data->serial = $vaccinationRow['serial_number'];
        $current_data->expiration = $vaccinationRow['expiration_date'];

        return $current_data;
    } catch (Exception $e) {
        echo 'Caught exception on assignCustomVaccinationData():', $e->getMessage(), '\n';
        exit(1);
    }
}

function processCustomHealthData($mysqli, $userID, $healthIDs, $row, $current_data)
{
    try {
        $all_data = [];
        $healthQuery = prepareCustomHealthQuery($mysqli, $healthIDs, $userID);
        $healthResultInner = $healthQuery->get_result();

        if ($healthResultInner === false) {
            die("Failed to get health result: (" . $healthQuery->errno . ") " . $healthQuery->error);
        }

        while ($healthRow = $healthResultInner->fetch_assoc()) {
            $current_data_inner = clone $current_data;
            $current_data_inner = assignCustomHealthData($current_data_inner, $healthRow, $row['health_id']);
            array_push($all_data, $current_data_inner);
        }

        return $all_data;
    } catch (Exception $e) {
        echo 'Caught exception on processCustomHealthData():', $e->getMessage(), '\n';
        exit(1);
    }
}

function processCustomVaccinationData($mysqli, $userID, $vaccinationIDs, $row, $current_data)
{
    try {
        $all_data = [];
        $vaccinationQuery = prepareCustomVaccinationQuery($mysqli, $vaccinationIDs, $userID);
        $vaccinationResultInner = $vaccinationQuery->get_result();

        if ($vaccinationResultInner === false) {
            die("Failed to get result: (" . $vaccinationQuery->errno . ") " . $vaccinationQuery->error);
        }

        while ($vaccinationRow = $vaccinationResultInner->fetch_assoc()) {
            $current_data_inner = clone $current_data;
            $current_data_inner = assignCustomVaccinationData($current_data_inner, $vaccinationRow, $row['vaccination_id']);
            array_push($all_data, $current_data_inner);
        }

        return $all_data;
    } catch (Exception $e) {
        echo 'Caught exception on processCustomVaccinationData():', $e->getMessage(), '\n';
        exit(1);
    }
}

function processCustomData($mysqli, $userID, $row)
{
    try {
        $current_data = new Health();
        $current_data->animal_id = $row['Animal'];
        $current_data->date = $row['Date'];
        $current_data->weight = $row['Weight'];
        $all_data = [];

        if ($row['health_id']) {
            $healthIDs = array_map('intval', explode(",", $row['health_id']));
            $all_data = array_merge($all_data, processCustomHealthData($mysqli, $userID, $healthIDs, $row, $current_data));
        }

        if ($row['vaccination_id']) {
            $vaccinationIDs = array_map('intval', explode(",", $row['vaccination_id']));
            $all_data = array_merge($all_data, processCustomVaccinationData($mysqli, $userID, $vaccinationIDs, $row, $current_data));
        }

        if (!$row['health_id'] && !$row['vaccination_id']) {
            array_push($all_data, $current_data);
        }

        return $all_data;
    } catch (Exception $e) {
        echo 'Caught exception on processCustomData():', $e->getMessage(), '\n';
        exit(1);
    }
}

function processCustomResult($mysqli, $userID, $healthResult)
{
    try {
        $all_data = [];

        while ($row = $healthResult->fetch_assoc()) {
            $all_data = array_merge($all_data, processCustomData($mysqli, $userID, $row));
        }

        return $all_data;
    } catch (Exception $e) {
        $errorMessage = [
            'error' => 'Caught exception on processCustomResult()',
            'message' => $e->getMessage()
        ];
        echo json_encode($errorMessage);
        exit(1);
    }
}


function customProcessingReport($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID)
{
    try {
        // $healthString =
        //     "SELECT
        //     A.`id`,
        //     A.`primary_id` AS 'Animal',
        //     EC.`event_datetime` AS 'Date',
        //     EC.`mature_weight` AS 'Weight',
        //     GROUP_CONCAT(DISTINCT EH.`id`) AS 'health_id',
        //     GROUP_CONCAT(DISTINCT EV.`id`) AS 'vaccination_id'
        //     FROM event_custom AS EC
        //     JOIN animals AS A
        //     ON EC.`animal_id` = A.`id`
        //     LEFT JOIN `group_animals` AS GA
        //     ON GA.`animal_id` = A.`id`
        //     LEFT JOIN `groups` AS G
        //     ON G.`id` = GA.`group_id`
        //     JOIN `statuses` AS ST
        //     ON ST.`id` = A.`status_id`
        //     LEFT JOIN `event_health_mapping` AS EHCM
        //     ON EC.`id` = EHCM.`event_id`
        //     LEFT JOIN `event_health` AS EH
        //     ON EH.`id` = EHCM.`health_id`
        //     LEFT JOIN `event_vaccination_mapping` AS EVCM
        //     ON EC.`id` = EVCM.`event_id`
        //     LEFT JOIN `event_vaccination` AS EV
        //     ON EV.`id` = EVCM.`vaccination_id`";

        $healthString = "SELECT
            A.`id`,
            A.`primary_id` AS 'Animal',
            EC.`event_datetime` AS 'Date',
            EC.`weight` AS 'Weight',
            GROUP_CONCAT(DISTINCT EH.`id`) AS 'health_id',
            GROUP_CONCAT(DISTINCT EV.`id`) AS 'vaccination_id'
            FROM `event_custom` AS EC
            JOIN `animals` AS A ON EC.`animal_id` = A.`id`
            JOIN `weight_types` AS WT ON EC.`weight_id` = WT.`id` AND WT.`id` = 3
            LEFT JOIN `group_animals` AS GA ON GA.`animal_id` = A.`id`
            LEFT JOIN `groups` AS G ON G.`id` = GA.`group_id`
            JOIN `statuses` AS ST ON ST.`id` = A.`status_id`
            LEFT JOIN `event_health_mapping` AS EHCM ON EC.`id` = EHCM.`event_id`
            LEFT JOIN `event_health` AS EH ON EH.`id` = EHCM.`health_id`
            LEFT JOIN `event_vaccination_mapping` AS EVCM ON EC.`id` = EVCM.`event_id`
            LEFT JOIN `event_vaccination` AS EV ON EV.`id` = EVCM.`vaccination_id`
            WHERE EC.`weight_id` = 3";

        // Initialize the parameters and conditions arrays
        $where = array();
        $params = array();
        $types = "";

        // Build the dynamic WHERE clause based on the function parameters
        if ($startDate) {
            $where[] = "EC.`event_datetime` >= ?";
            $params[] = $startDate;
            $types .= "s";
        }
        if ($endDate) {
            $where[] = "EC.`event_datetime` <= ?";
            $params[] = $endDate;
            $types .= "s";
        }
        if ($IDs) {
            $whereIn = implode(',', array_fill(0, count($IDs), '?'));
            $where[] = ($idType === 'groups' ? "G" : "A") . ".`id` IN ($whereIn)";
            $params = array_merge($params, $IDs);
            $types .= str_repeat('i', count($IDs));
        }
        if ($statusID) {
            $statusID = implode(',', (array) $statusID); // Ensure $statusID is treated as an array
            $where[] = "A.`status_id` IN ($statusID)";
            // Note: $statusID is included directly in the query without binding as a parameter
        }
        $where[] = "EC.`user_id` = ?";
        $params[] = $userID;
        $types .= "s";

        // Append the WHERE conditions to the base query
        if (!empty($where)) {
            $healthString .= " AND " . implode(" AND ", $where);
        }

        // Complete the query with GROUP BY and ORDER BY clauses
        // $healthString .= " GROUP BY A.`id` ORDER BY EC.`event_datetime` DESC";
        $healthString .= " GROUP BY A.`id`, A.`primary_id`, EC.`event_datetime`, EC.`weight`";

        // Prepare the SQL statement
        $query = $mysqli->prepare($healthString);
        if ($query === false) {
            throw new Exception('prepare() failed: ' . htmlspecialchars($mysqli->error));
        }

        // Bind the parameters to the SQL statement
        if ($types) { // Make sure there are parameters to bind
            $query->bind_param($types, ...$params);
        }

        // Execute the query
        if ($query->execute()) {
            $healthResult = $query->get_result();
            $query->close();

            // Process the result as needed
            $all_data = processCustomResult($mysqli, $userID, $healthResult);

            // Output the data in JSON format
            echo json_encode($all_data);
        } else {
            throw new Exception('execute() failed: ' . htmlspecialchars($query->error));
        }
    } catch (Exception $e) {
        $errorMessage = [
            'error' => 'Caught exception on customProcessingReport()',
            'message' => $e->getMessage()
        ];
        echo json_encode($errorMessage);
        exit(1);
    }
}


function pedigreeReportQuery($mysqli, $userID, $reportType, $selectedItem)
{
    try {
        if ($reportType !== "ped") {
            echo json_encode('Invalid Health Report Type!');
            exit(1);
        }

        $pedigreeReportQuery = $mysqli->prepare(
            "SELECT
                A.`id`,
                A.`primary_id` AS 'Animal',
                S.`primary_id` AS 'Sire',
                D.`primary_id` AS 'Dam',
                PS.`primary_id` AS 'Paternal_grandsire',
                PD.`primary_id` AS 'Paternal_grandam',
                MS.`primary_id` AS 'Maternal_grandsire',
                MD.`primary_id` AS 'Maternal_grandam',
                PPS.`primary_id` AS 'Sire_paternal_grandsire',
                PPD.`primary_id` AS 'Sire_paternal_grandam',
                PMS.`primary_id` AS 'Sire_maternal_grandsire',
                PMD.`primary_id` AS 'Sire_maternal_grandam',
                MPS.`primary_id` AS 'Dam_paternal_grandsire',
                MPD.`primary_id` AS 'Dam_paternal_grandam',
                MMS.`primary_id` AS 'Dam_maternal_grandsire',
                MMD.`primary_id` AS 'Dam_maternal_grandam'
            FROM animals AS A
            LEFT JOIN animals AS S ON S.`id` = A.`sire`
            LEFT JOIN animals AS D ON D.`id` = A.`dam`
            LEFT JOIN animals AS PS ON PS.`id` = S.`sire`
            LEFT JOIN animals AS PD ON PD.`id` = S.`dam`
            LEFT JOIN animals AS MS ON MS.`id` = D.`sire`
            LEFT JOIN animals AS MD ON MD.`id` = D.`dam`
            LEFT JOIN animals AS PPS ON PPS.`id` = PS.`sire`
            LEFT JOIN animals AS PPD ON PPD.`id` = PS.`dam`
            LEFT JOIN animals AS PMS ON PMS.`id` = PD.`sire`
            LEFT JOIN animals AS PMD ON PMD.`id` = PD.`dam`
            LEFT JOIN animals AS MPS ON MPS.`id` = MS.`sire`
            LEFT JOIN animals AS MPD ON MPD.`id` = MS.`dam`
            LEFT JOIN animals AS MMS ON MMS.`id` = MD.`sire`
            LEFT JOIN animals AS MMD ON MMD.`id` = MD.`dam`
            WHERE A.`user_id` = ?
            AND A.`id` = ?
            -- The COALESCE() function returns the first non-null value in a list.
            AND (COALESCE(A.`sire`, A.`dam`) IS NOT NULL)"
        );

        if (!$pedigreeReportQuery || !$pedigreeReportQuery->bind_param('si', $userID, $selectedItem->id) || !$pedigreeReportQuery->execute()) {
            echo json_encode($mysqli->error);
            exit(1);
        }

        $pedigreeResult = $pedigreeReportQuery->get_result();
        $pedigreeReportQuery->close();

        $result = array();
        while ($row = $pedigreeResult->fetch_assoc()) {
            $result[] = $row;
        }

        echo json_encode($result);
    } catch (Exception $e) {
        $errorMessage = [
            'error' => 'Caught exception on pedigreeReportQuery()',
            'message' => $e->getMessage()
        ];
        echo json_encode($errorMessage);
        exit(1);
    }
}


function growthReportQuery($mysqli, $userID, $startDate, $endDate, $weightType, $IDs, $idType, $statusID)
{
    try {
        // $baseQuery = "SELECT
        //     T1.`Animal` AS 'Animal',
        //     T1.`Sex` AS 'Sex',
        //     T1.`Sire` AS 'Sire',
        //     T1.`Birth_Weight` AS 'Birth_Weight',
        //     T1.`Birth_Date` AS 'Birth_Date',
        //     T1.`Weaning_Weight` AS 'Weaning_Weight',
        //     T1.`Weaning_Date` AS 'Weaning_Date',
        //     T2.`Yearling_Weight` AS 'Yearling_Weight',
        //     T2.`Yearling_Date` AS 'Yearling_Date'
        // FROM
        //     (SELECT
        //         A.`id` AS 'Animal_ID',
        //         A.`primary_id` AS 'Animal',
        //         S.`name` AS 'Sex',
        //         A1.`primary_id` AS 'Sire',
        //         A.`weight` AS 'Birth_Weight',
        //         A.`date_birth` AS 'Birth_Date',
        //         EC.`weaning_weight` AS 'Weaning_Weight',
        //         EC.`event_datetime` AS 'Weaning_Date',
        //         EC.`user_id`
        //     FROM `event_custom` AS EC
        //     JOIN `animals` AS A ON A.`id` = EC.`animal_id`
        //     JOIN `sexes` AS S ON S.`id` = A.`sex_id`
        //     LEFT JOIN `animals` AS A1 ON A1.`id` = A.`sire`
        //     WHERE EC.`user_id` = ?
        //     AND EC.`weaning_weight` IS NOT NULL) AS T1
        // JOIN
        //     (SELECT
        //         EC.`animal_id` AS 'Animal_ID',
        //         EC.`yearling_weight` AS 'Yearling_Weight',
        //         EC.`event_datetime` AS 'Yearling_Date'
        //     FROM `event_custom` AS EC
        //     WHERE EC.`user_id` = ?
        //     AND EC.`yearling_weight` IS NOT NULL) AS T2
        // ON T1.Animal_ID = T2.Animal_ID";

        $baseQuery = "SELECT
            A.`id` AS 'Animal_ID',
            A.`primary_id` AS 'Animal',
            S.`name` AS 'Sex',
            A.`weight` AS 'Birth_Weight',
            A.`date_birth` AS 'Birth_Date',
            EC.`weight` AS 'Weight',
            WT.`description` AS 'Weight_Type',
            EC.`event_datetime` AS 'Date',
            -- Calculate the difference in years, but return NULL if date_birth is NULL
            CASE 
                WHEN A.`date_birth` IS NOT NULL THEN TIMESTAMPDIFF(YEAR, A.`date_birth`, EC.`event_datetime`)
                ELSE NULL 
            END AS 'Age_Years',
            -- Calculate the difference in days, but return NULL if date_birth is NULL
            CASE 
                WHEN A.`date_birth` IS NOT NULL THEN TIMESTAMPDIFF(DAY, A.`date_birth`, EC.`event_datetime`) - (TIMESTAMPDIFF(YEAR, A.`date_birth`, EC.`event_datetime`) * 365)
                ELSE NULL 
            END AS 'Age_Days'
        FROM `event_custom` AS EC
        JOIN `animals` AS A ON A.`id` = EC.`animal_id`
        JOIN `sexes` AS S ON S.`id` = A.`sex_id`
        JOIN `weight_types` AS WT ON EC.`weight_id` = WT.`id`";

        if ($idType == 'groups') {
            $baseQuery .= " JOIN `group_animals` AS GA ON GA.`animal_id` = A.`id`
                            JOIN `groups` AS G ON G.`id` = GA.`group_id`";
        }

        $baseQuery .= " WHERE EC.`user_id` = ?";
        $params = [$userID];
        $paramTypes = 's';

        // Check if a specific weight ID is provided, otherwise use the default set
        if (is_numeric($weightType)) {
            // If a single numeric weightType is provided, use it
            $baseQuery .= " AND EC.`weight_id` = ?";
            $params[] = (int)$weightType;
            $paramTypes .= 'i';
        } else {
            // If the default set of weight IDs is provided, prepare to use them in the IN clause
            $weightTypeIds = explode(',', $weightType);
            $weightTypePlaceholders = implode(',', array_fill(0, count($weightTypeIds), '?'));
            $baseQuery .= " AND EC.`weight_id` IN ($weightTypePlaceholders)";
            foreach ($weightTypeIds as $wtId) {
                $params[] = (int)$wtId;
                $paramTypes .= 'i';
            }
        }

        // Ensure that $statusID is an array
        $statusID = (array) $statusID;
        // Ensure that $IDs is an array
        $IDs = (array) $IDs;

        if (!empty($statusID)) {
            $statusPlaceholders = implode(',', array_fill(0, count($statusID), '?'));
            $baseQuery .= " AND A.`status_id` IN ($statusPlaceholders)";
            $params = array_merge($params, $statusID);
            $paramTypes .= str_repeat('i', count($statusID));
        }

        if ($startDate) {
            $baseQuery .= " AND EC.`event_datetime` >= ?";
            $params[] = $startDate;
            $paramTypes .= 's';
        }

        if ($endDate) {
            $baseQuery .= " AND EC.`event_datetime` <= ?";
            $params[] = $endDate;
            $paramTypes .= 's';
        }

        if (!empty($IDs)) {
            $idPlaceholders = implode(',', array_fill(0, count($IDs), '?'));
            if ($idType == 'groups') {
                $baseQuery .= " AND G.`id` IN ($idPlaceholders)";
            } else {
                $baseQuery .= " AND A.`id` IN ($idPlaceholders)";
            }
            $params = array_merge($params, $IDs);
            $paramTypes .= str_repeat('i', count($IDs));
        }

        $baseQuery .= " ORDER BY EC.`event_datetime` DESC";

        $query = $mysqli->prepare($baseQuery);
        if ($query === false) {
            throw new Exception('Prepare Error: ' . $mysqli->error);
        }

        $query->bind_param($paramTypes, ...$params);
        if (!$query->execute()) {
            throw new Exception($mysqli->error);
        }

        $growthResult = $query->get_result();
        $query->close();

        $result = [];
        while ($row = $growthResult->fetch_assoc()) {
            $result[] = $row;
        }
        $growthResult->free();

        echo json_encode($result);
    } catch (Exception $e) {
        $errorMessage = [
            'error' => 'Caught exception on growthReportQuery()',
            'message' => $e->getMessage()
        ];
        echo json_encode($errorMessage);
        exit(1);
    }
}


function pregnancyCheckReportQuery($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID)
{
    try {
        $baseQuery = "SELECT
                A.`id`,
                A.`primary_id` as 'Animal',
                EC.`event_datetime` AS 'Date',
                EC.`weight` AS 'Weight',
                WT.`description` AS 'Weight_Type',
                EC.`bcs` AS 'Body_Condition_Score',
                EC.`chute_docility_score` As 'Chute_Docility_Score',
                EC.`foot_angle_score` AS 'Foot_Angle_Score',
                EC.`claw_set_score` AS 'Claw_Set_Score',
                EC.`pregnancy_status` AS 'Pregnancy_Status',
                EC.`gestation_length` AS 'Gestation_Length',
                EC.`gl_units` AS 'Gestation_Length_Units',
                EC.`notes` AS 'Notes'
                FROM `animals` AS A
                JOIN `event_custom` AS EC ON EC.`animal_id` = A.`id`
                JOIN `statuses` AS ST ON ST.`id` = A.`status_id`
                JOIN `weight_types` AS WT ON WT.`id` = EC.`weight_id`";

        // This section handles the JOIN based on ID type
        if ($idType == 'groups') {
            $baseQuery .= " JOIN `group_animals` AS GA ON GA.`animal_id` = A.`id`
                            JOIN `groups` AS G ON G.`id` = GA.`group_id`";
        }

        $baseQuery .= " WHERE A.`user_id` = ?";
        $params = [$userID];
        $paramTypes = 's';

        // Ensure that $statusID is an array
        $statusID = (array) $statusID;
        // Ensure that $IDs is an array
        $IDs = (array) $IDs;

        if (!empty($statusID)) {
            $statusPlaceholders = implode(',', array_fill(0, count($statusID), '?'));
            $baseQuery .= " AND A.`status_id` IN ($statusPlaceholders)";
            $params = array_merge($params, $statusID);
            $paramTypes .= str_repeat('i', count($statusID));
        }

        // Handle date condition
        if ($startDate) {
            $baseQuery .= " AND EC.`event_datetime` >= ?";
            $params[] = $startDate;
            $paramTypes .= 's';
        }
        if ($endDate) {
            $baseQuery .= " AND EC.`event_datetime` < ?";
            $params[] = $endDate;
            $paramTypes .= 's';
        }

        // Handle ID type condition
        if (!empty($IDs)) {
            $idPlaceholders = implode(',', array_fill(0, count($IDs), '?'));
            if ($idType == 'groups') {
                $baseQuery .= " AND G.`id` IN ($idPlaceholders)";
            } else {
                $baseQuery .= " AND A.`id` IN ($idPlaceholders)";
            }
            $params = array_merge($params, $IDs);
            $paramTypes .= str_repeat('i', count($IDs));
        }

        $baseQuery .= " ORDER BY EC.`event_datetime` DESC";

        $query = $mysqli->prepare($baseQuery);
        if ($query === false) {
            throw new Exception('Prepare Error: ' . $mysqli->error);
        }

        $query->bind_param($paramTypes, ...$params);
        if (!$query->execute()) {
            throw new Exception($mysqli->error);
        }

        $pregnancyResult = $query->get_result();
        $query->close();

        $result = array();
        while ($row = $pregnancyResult->fetch_assoc()) {
            $result[] = $row;
        }
        $pregnancyResult->free();

        echo json_encode($result);
    } catch (Exception $e) {
        $errorMessage = [
            'error' => 'Caught exception on pregnancyCheckReportQuery()',
            'message' => $e->getMessage()
        ];
        echo json_encode($errorMessage);
        exit(1);
    }
}


function birthReportQuery($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID)
{
    try {
        // Base query to fetch details of birth report
        $queryString = "SELECT
            A.`id`,
            A.`primary_id` AS 'Calf',
            G.`group_name` AS 'Group',
            S.`type` AS 'Calf_Sex',
            ST.`name` AS 'Status',
            A.`weight` AS 'Birth_Weight',
            A.`date_birth` AS 'Birth_Date',
            A1.`primary_id` AS 'Sire',
            A2.`primary_id` AS 'Dam'
            FROM animals AS A
            LEFT JOIN animals AS A1
            ON A1.`id` = A.`sire`
            LEFT JOIN animals AS A2
            ON A2.`id` = A.`dam`
            JOIN sexes AS S
            ON S.`id` = A.`sex_id`
            JOIN `statuses` AS ST
            ON ST.`id` = A.`status_id`
            JOIN `group_animals` AS GA
            ON GA.`animal_id` = A.`id`
            JOIN `groups` AS G
            ON G.`id` = GA.`group_id`
            WHERE A.`status_id` IN (" . $statusID . ")
            AND A.`user_id` = ?";

        // Initialize parameters and types
        $params = [$userID];
        $paramTypes = 's';

        // Appending to WHERE clause for groups or animals
        if (($idType == 'groups' || $idType == 'animals') && !empty($IDs)) {
            $idPlaceholders = implode(',', array_fill(0, count($IDs), '?'));
            $queryString .= ($idType == 'groups') ? " AND G.`id` IN ($idPlaceholders)" : " AND A.`id` IN ($idPlaceholders)";
            $params = array_merge($params, $IDs);
            // 'i' for integers
            $paramTypes .= str_repeat('i', count($IDs));
        }

        // Filter by start date
        if ($startDate) {
            $queryString .= " AND A.`date_birth` >= ?";
            $params[] = $startDate;
            $paramTypes .= 's';
        }

        // Filter by end date
        if ($endDate) {
            $queryString .= " AND A.`date_birth` < ?";
            $params[] = $endDate;
            $paramTypes .= 's';
        }

        // Adding the ORDER BY clause at the end
        $queryString .= " ORDER BY A.`primary_id` ASC";

        // Preparing the query
        $query = $mysqli->prepare($queryString);

        // Binding parameters
        $query->bind_param($paramTypes, ...$params);

        // Executing the query and fetching results
        if ($query->execute()) {
            $birthResult = $query->get_result();
            $query->close();

            $result = array();

            while ($row = $birthResult->fetch_assoc()) {
                $result[] = $row;
            }

            echo json_encode($result);
        } else {
            // Throwing an exception in case of query execution error
            throw new Exception("Query execution error: " . $mysqli->error);
        }
    } catch (Exception $e) {
        $errorMessage = [
            'error' => 'Caught exception on birthReportQuery()',
            'message' => $e->getMessage()
        ];
        echo json_encode($errorMessage);
        exit(1);
    }
}


function buildEventQuery($baseString, $statusID, $userID, $startDate, $endDate, $IDs, $idType)
{
    try {
        // Initialize parameters array
        $params = array();

        // Create the basic WHERE clause filtering by status ID and user ID
        $whereClause = " WHERE A.`status_id` = ? AND A.`user_id` = ?";
        array_push($params, $statusID, $userID);

        // Append date filters if start and/or end date is provided
        if ($startDate && $endDate) {
            $whereClause .= " AND `date` BETWEEN ? AND ?";
            array_push($params, $startDate, $endDate);
        } elseif ($startDate) {
            $whereClause .= " AND `date` >= ?";
            array_push($params, $startDate);
        } elseif ($endDate) {
            $whereClause .= " AND `date` <= ?";
            array_push($params, $endDate);
        }

        /*
        * If $IDs is not empty, append additional filter based on $idType ('groups' or 'animals')
        * If $IDs is empty, this block is skipped, so no additional filtering is applied
        */
        if (!empty($IDs)) {
            $idPlaceholders = implode(',', array_fill(0, count($IDs), '?'));
            if ($idType === 'groups') {
                $whereClause .= " AND G.`id` IN (" . $idPlaceholders . ")";
            } elseif ($idType === 'animals') {
                $whereClause .= " AND A.`id` IN (" . $idPlaceholders . ")";
            }
            $params = array_merge($params, $IDs);
        }

        // Append the ordering clause
        $orderClause = " ORDER BY `date` DESC, A.`primary_ID`, G.`group_name`";

        // Combine the base string, WHERE clause, and ORDER BY clause to form the final query
        $query = $baseString . $whereClause . $orderClause;

        // Return the query string and parameters array
        return array('query' => $query, 'params' => $params);
    } catch (Exception $e) {
        echo 'Caught exception on buildEventQuery(): ', $e->getMessage(), '\n';
        exit(1);
    }
}


// Initial function to generate health event report (calls buildEventQuery() to help build query)
function healthEventReport($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID)
{
    try {
        $healthString = "SELECT
        A.`id`,
        A.`primary_ID` AS 'Animal',
        G.`group_name` AS 'Group',
        GC.`category_name` AS 'Category',
        H.`id` AS 'Event_ID',
        H.`date` AS 'Event_Date',
        H.`weight` AS 'Weight',
        BS.`system_name` AS 'System',
        BSC.`condition_name` AS 'Condition',
        H.`product` AS 'Product_Used',
        H.`product_other` AS 'Product_Other',
        H.`volume` AS 'Volume',
        U.`description` AS 'Units',
        RA.`description` AS 'Route',
        TR.`name` AS 'Reason',
        H.`notes` AS 'Notes',
        'Health' as 'Event_Type'
        FROM `animals` AS A
        INNER JOIN `event_health` AS H
            ON H.`animal_id` = A.`id`
        LEFT JOIN `units` AS U
            ON U.`id` = H.`units_id`
        LEFT JOIN `route_of_administration` AS RA
            ON RA.`id` = H.`roa_id`
        LEFT JOIN `treatment_reasons` AS TR
            ON TR.`id` = H.`reason`
        JOIN `body_systems` AS BS
            ON BS.`id` = H.`problem_bs`
        JOIN `body_system_conditions` AS BSC
            ON BSC.`id` = H.`problem_bsc`
        JOIN `group_animals` AS GA
            ON GA.`animal_id` = A.`id`
        JOIN `groups` AS G
            ON G.`id` = GA.`group_id`
        JOIN `group_categories` AS GC
            ON GC.`id` = G.`category_id`
        JOIN `statuses` AS ST
            ON ST.`id` = A.`status_id`";

        $vaccinationString = "SELECT
            A.`id`,
            A.`primary_ID`,
            G.`group_name`,
            GC.`category_name`,
            V.`id`,
            V.`date`,
            NULL as 'Weight',
            NULL as 'System',
            NULL as 'Condition',
            V.`vaccine`,
            V.`vaccine_other`,
            V.`volume`,
            U.`description`,
            RA.`description`,
            NULL as 'Reason',
            NULL as 'Notes',
            'Vaccination'
            FROM `animals` AS A
            INNER JOIN `event_vaccination` AS V
                ON V.`animal_id` = A.`id`
            LEFT JOIN `units` AS U
                ON U.`id` = V.`units_id`
            LEFT JOIN `route_of_administration` AS RA
                ON RA.`id` = V.`roa_id`
            JOIN `group_animals` AS GA
                ON GA.`animal_id` = A.`id`
            JOIN `groups` AS G
                ON G.`id` = GA.`group_id`
            JOIN `group_categories` AS GC
                ON GC.`id` = G.`category_id`
            JOIN `statuses` AS ST
                ON ST.`id` = A.`status_id`";

        $healthQueryData = buildEventQuery($healthString, $statusID, $userID, $startDate, $endDate, $IDs, $idType);
        $vaccinationQueryData = buildEventQuery($vaccinationString, $statusID, $userID, $startDate, $endDate, $IDs, $idType);

        $queryString = "(" . $healthQueryData['query'] . ") UNION ALL (" . $vaccinationQueryData['query'] . ")";
        $params = array_merge($healthQueryData['params'], $vaccinationQueryData['params']);

        $stmt = $mysqli->prepare($queryString);
        if ($stmt === false) {
            die('prepare() failed: ' . htmlspecialchars($mysqli->error));
        }

        $types = str_repeat('s', count($params));
        $stmt->bind_param($types, ...$params);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $stmt->close();

            $output = array();
            while ($row = $result->fetch_assoc()) {
                $output[] = $row;
            }

            echo json_encode($output);
        } else {
            echo json_encode("SQL Error: " . $mysqli->error);
            exit(1);
        }
    } catch (Exception $e) {
        echo 'Caught exception on healthEventReport():', $e->getMessage(), '\n';
        exit(1);
    }
}

function cowProgenyReport($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID)
{
    try {
        // Building the base query for cow progeny report
        $cowProgenyQuery = "SELECT
            A1.`id` AS 'Calf_ID',
            A1.`primary_id` AS 'Calf',
            A.`id` AS 'Cow_ID',
            A.`primary_id` AS 'Cow',
            G.`group_name` AS 'Cow_Group',
            S.`type` AS 'Calf_Sex',
            A1.`weight` AS 'Calf_Birth_Weight',
            A1.`date_birth` AS 'Calf_Birth_Date',
            G1.`group_name` AS 'Calf_Group'
            FROM animals AS A
            JOIN `group_animals` AS GA
            ON GA.`animal_id` = A.`id`
            JOIN `groups` AS G
            ON G.`id` = GA.`group_id`
            JOIN animals AS A1
            ON A1.`dam` = A.`id`
            JOIN `group_animals` AS GA1
            ON GA1.`animal_id` = A1.`id`
            JOIN `groups` AS G1
            ON G1.`id` = GA1.`group_id`
            JOIN sexes AS S
            ON S.`id` = A1.`sex_id`
            WHERE A.`status_id` IN (" . $statusID . ")
            AND A.`user_id` = ?";

        // Initializing the parameters
        $params = [$userID];
        $paramTypes = 's';

        // Appending to WHERE clause for groups or animals
        if (($idType == 'groups' || $idType == 'animals') && !empty($IDs)) {
            $idPlaceholders = implode(',', array_fill(0, count($IDs), '?'));
            $cowProgenyQuery .= ($idType == 'groups') ? " AND G.`id` IN ($idPlaceholders)" : " AND A.`id` IN ($idPlaceholders)";
            $params = array_merge($params, $IDs);
            $paramTypes .= str_repeat('i', count($IDs)); // 'i' for integers
        }

        // Appending to WHERE clause if (calf birth) start date is set
        if ($startDate) {
            $cowProgenyQuery .= " AND A1.`date_birth` >= ?";
            $params[] = $startDate;
            $paramTypes .= 's';
        }

        // Appending to WHERE clause if (calf birth) end date is set
        if ($endDate) {
            $cowProgenyQuery .= " AND A1.`date_birth` < ?";
            $params[] = $endDate;
            $paramTypes .= 's';
        }

        // Adding the ORDER BY clause at the end
        $cowProgenyQuery .= " ORDER BY A.`primary_id` ASC";

        // Preparing the query
        $query = $mysqli->prepare($cowProgenyQuery);

        // Binding parameters
        $query->bind_param($paramTypes, ...$params);

        // Executing the query and fetching the result
        if ($query->execute()) {
            $dateResult = $query->get_result();
            $query->close();

            // Initializing result array
            $result = array();

            // Fetching each row and appending to result array
            while ($row = $dateResult->fetch_assoc()) {
                $result[] = $row;
            }

            // Returning the result array as JSON
            echo json_encode($result);
        } else {
            // Query execution error
            echo json_encode($mysqli->error);
            exit(1);
        }
    } catch (Exception $e) {
        // Handling exceptions
        echo 'Caught exception on cowProgenyReport(): ', $e->getMessage(), '\n';
        exit(1);
    }
}


if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $message = 'Invalid request method!';
    goto error;
}

$data = file_get_contents('php://input');
$request = json_decode($data);

if (!$request || count($request) < 4) {
    $message = 'Error: No request data for reports!';
    goto error;
}

require('connection.php');

$reportType = $mysqli->real_escape_string($request[0]);
$userID = $mysqli->real_escape_string($request[1]);
$loggedIn = $request[2];
$selectedItem = $request[3];

$reportTypesA = ['ped'];
$reportTypesB = ['healthEvent', 'customProcessing', 'birth', 'pcr', 'weaning', 'ccr', 'cpr', 'growth', 'air', 'bior', 'nsbr'];

if (in_array($reportType, $reportTypesA)) {
    switch ($reportType) {
        case 'ped':
            pedigreeReportQuery($mysqli, $userID, $reportType, $selectedItem);
            break;
        default:
            $message = 'Error: No report type for reports (B)!';
            goto error;
    }
} else if (in_array($reportType, $reportTypesB)) {
    $startDate = NULL;
    $endDate = NULL;
    $IDs = null;
    $idType = null;
    $groupIDs = [];
    $animalIDs = [];

    if (isset($request[3]->data[0])) {
        $startDate = $mysqli->real_escape_string($request[3]->data[0]);
    }

    if (isset($request[3]->data[1])) {
        $endDate = $mysqli->real_escape_string($request[3]->data[1]);

        try {
            $endDateObject = new DateTime($endDate);
            $endDateObject->modify('+1 day');
            $endDate = $endDateObject->format('Y-m-d');
        } catch (Exception $e) {
            die('Error: ' . $e->getMessage());
        }
    }

    if (isset($request[3]->data[2])) {
        $statusID = $mysqli->real_escape_string($request[3]->data[2]);
    } else {
        $statusID = '1,2';
    }

    if (isset($request[3]->data[3]) && $request[3]->data[3] !== '0' && $request[3]->data[3] !== '') {
        $weightType = $mysqli->real_escape_string($request[3]->data[3]);
    } else {
        $weightType = '1,2,3,4,5,6';
    }

    // Check to see if the user selected groups (cannot select groups and animals at the same time)
    if (isset($request[3]->groups)) {
        foreach ($request[3]->groups as $g) {
            $groupIDs[] = intval($g->id);
        }

        if (!empty($groupIDs)) {
            $IDs = $groupIDs;  // now $IDs is an array
            $idType = 'groups';
        }
    }

    // Check to see if the user selected animals (cannot select groups and animals at the same time)
    if (isset($request[3]->animals)) {
        foreach ($request[3]->animals as $a) {
            $animalIDs[] = intval($a->id);
        }

        if (!empty($animalIDs)) {
            $IDs = $animalIDs;  // now $IDs is an array
            $idType = 'animals';
        }
    }

    switch ($reportType) {
        case 'healthEvent':
            healthEventReport($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID);
            break;
        case 'customProcessing':
            customProcessingReport($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID);
            break;
        case 'birth':
            birthReportQuery($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID);
            break;
        case 'pcr':
            pregnancyCheckReportQuery($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID);
            break;
        case 'weaning':
            weaningReportQuery($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID);
            break;
        case 'ccr':
            cowCalvingReportQuery($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID);
            break;
        case 'cpr':
            cowProgenyReport($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID);
            break;
        case 'growth':
            growthReportQuery($mysqli, $userID, $startDate, $endDate, $weightType, $IDs, $idType, $statusID);
            break;
        case 'air':
            breedingReportQuery($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID, $reportType);
            break;
        case 'nsbr':
            breedingReportQuery($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID, $reportType);
            break;
        case 'bior':
            bullInBullOutQuery($mysqli, $userID, $startDate, $endDate, $IDs, $idType, $statusID, $reportType);
            break;
        default:
            $message = 'Error: No report type for reports (A)!';
            goto error;
    }
} else {
    $message = 'Error: No report type for reports (A)!';
    goto error;
}

exit(0);

error:
echo json_encode($message);
exit(1);
